/**
 * PETALS - PETALS Services Platform. Copyright (c) 2005 EBM Websourcing,
 * http://www.ebmwebsourcing.com/
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version. This library is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 * ---------------------------------------------------------------------------
 * ---------------------------------------------------------------------------
 */

package org.ow2.petals.component.framework.api.notification;

import javax.jbi.messaging.MessageExchange;
import javax.jbi.messaging.MessagingException;

import org.ow2.easywsdl.wsdl.api.WSDLException;
import org.ow2.petals.component.framework.api.exception.PEtALSCDKException;
import org.w3c.dom.Document;

import com.ebmwebsourcing.wsstar.addressing.definition.api.EndpointReferenceType;
import com.ebmwebsourcing.wsstar.addressing.definition.api.WSAddressingException;
import com.ebmwebsourcing.wsstar.notification.definition.brokerednotification.api.RegisterPublisherResponse;
import com.ebmwebsourcing.wsstar.notification.definition.utils.WSNotificationException;
import com.ebmwebsourcing.wsstar.notification.extension.utils.WSNotificationExtensionException;

/**
 * The controller of the Broker into the bus
 * 
 * @author Frederic Gardes
 * 
 */
public interface NotificationBrokerController {
    /**
     * register the publisher on the broker
     * 
     * @throws WSNotificationException
     * @throws MessagingException
     * @throws PEtALSCDKException
     * @throws WSAddressingException
     * @throws WSNotificationExtensionException
     */
    public void registerPublisher() throws WSNotificationException, MessagingException,
            PEtALSCDKException, WSAddressingException, WSNotificationExtensionException;

    /**
     * Persist publisher registration ref and notification consumer ref in
     * notification context
     * 
     * @param response
     * @throws PEtALSCDKException
     */
    void processRegisterPublisherResponse(RegisterPublisherResponse response)
            throws PEtALSCDKException;

    /**
     * subscribe the consumer on the broker
     * 
     * @throws WSNotificationException
     * @throws MessagingException
     * @throws PEtALSCDKException
     * @throws WSAddressingException
     */
    public void subscribe() throws WSNotificationException, MessagingException, PEtALSCDKException,
            WSAddressingException;

    /**
     * unsubscribe the consumer on the broker
     * 
     * @throws WSNotificationException
     * @throws MessagingException
     * @throws PEtALSCDKException
     * @throws WSAddressingException
     */
    public void unsubscribe() throws WSNotificationException, MessagingException,
            PEtALSCDKException, WSAddressingException;

    /**
     * Get the current message for the consumer on the broker
     * 
     * @return the current message
     * @throws WSNotificationException
     * @throws MessagingException
     * @throws PEtALSCDKException
     * @throws WSAddressingException
     */
    public String getCurrentMessage() throws WSNotificationException, MessagingException,
            PEtALSCDKException, WSAddressingException;

    /**
     * Destroy the registration of the publisher on the broker
     * 
     * @throws WSNotificationExtensionException
     * @throws MessagingException
     * @throws PEtALSCDKException
     * 
     */
    public void destroyRegistration() throws WSNotificationException,
            WSNotificationExtensionException, WSAddressingException, MessagingException,
            PEtALSCDKException;

    /**
     * notify from the publisher to the broker
     * 
     * @param document
     *            The message document payload into the notification
     * @param topic
     *            The topic into the notification
     * @param subscriptionReference
     * @throws WSNotificationException
     * @throws WSNotificationExtensionException
     * @throws MessagingException
     * @throws PEtALSCDKException
     * @throws WSDLException
     * @throws WSAddressingException
     */
    public void notify(Document document, String topic, EndpointReferenceType subscriptionReference)
            throws WSNotificationException, MessagingException, PEtALSCDKException,
            WSAddressingException, WSDLException, WSNotificationExtensionException;

    /**
     * Manage the renew on a subscription from the consumer
     */
    public void renew();

    /**
     * Check if the JBI exchange is a brokered notification exchange
     * 
     * @param messageExchange
     *            the JBI exchange
     * @return true if the exchange is a brokered notification exchange, false
     *         else
     */
    public boolean isBrokeredNotification(MessageExchange messageExchange);

    /**
     * Set the notificationBrokerReferenceAddress
     * 
     * @param notificationBrokerReferenceAddress
     *            the notificationBrokerReferenceAddress to set
     */
    public void setNotificationBrokerReferenceAddress(String notificationBrokerReferenceAddress);

    /**
     * Set the publisherRegistrationManagerReferenceAddress
     * 
     * @param publisherRegistrationManagerReferenceAddress
     *            the publisherRegistrationManagerReferenceAddress to set
     */
    public void setPublisherRegistrationManagerReferenceAddress(
            String publisherRegistrationManagerReferenceAddress);

    /**
     * Set the subscriptionManagerReferenceAddress
     * 
     * @param subscriptionManagerReferenceAddress
     *            the subscriptionManagerReferenceAddress to set
     */
    public void setSubscriptionManagerReferenceAddress(String subscriptionManagerReferenceAddress);

    /**
     * Get the notificationProducer
     * 
     * @return the notificationProducer
     */
    public NotificationProducer getNotificationProducer();

    /**
     * Get the notificationConsumer
     * 
     * @return the notificationConsumer
     */
    public NotificationConsumer getNotificationConsumer();

    /**
     * Check if the address is the default cdk address
     * 
     * @param address
     *            The address to check
     * @return true if the address is the default cdk address
     */
    public boolean isInternalNotification(String address);

    public Transformer getTransformer();
}
