/**
 * PETALS - PETALS Services Platform. Copyright (c) 2007 EBM Websourcing,
 * http://www.ebmwebsourcing.com/
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version. This library is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 * -------------------------------------------------------------------------
 * $Id$
 * -------------------------------------------------------------------------
 */

package org.ow2.petals.component.framework.api.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLConnection;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Map.Entry;

import org.ow2.petals.component.framework.api.exception.PEtALSCDKException;

/**
 * Utility to handle a CDK properties file.
 * 
 * @author ofabre - EBM Websourcing
 * @since 1.0
 */
public class PropertiesUtil {

    /**
     * <p>
     * Load a {@link Properties} object from a properties file path. The file
     * path can be relative path based on Petals root path or a valid URL.
     * </p>
     * <p>
     * Exemples : <br/>
     * Relative path : /conf/external.properties <br/>
     * Absolute path : file:///C:/conf/external.properties <br/>
     * URL : ftp://user:pwd@host:port/external.properties
     * </p>
     * 
     * @param propertiesFilePath
     *            The file path can be relative path based on Petals root path
     *            or a valid URL
     * @return a {@link Properties} object fill with properties value found in
     *         the targeted file
     * @throws PEtALSCDKException
     *             if error occurs when reading external properties file
     */
    public final static Properties loadProperties(final String propertiesFilePath)
    throws PEtALSCDKException {
        final Properties properties = new Properties();

        if (propertiesFilePath != null) {
            InputStream propFileStream = null;
            try {
                propFileStream = loadPropertiesFileFromUrl(new URL(propertiesFilePath));
            } catch (final MalformedURLException e) {
                propFileStream = loadPropertiesFileFromRelativePath(propertiesFilePath);
            }
            if (propFileStream != null) {
                try {
                    properties.load(propFileStream);
                } catch (final IOException e) {
                    throw new PEtALSCDKException("Cannot find properties at location : "
                            + propertiesFilePath);
                }
            } else {
                throw new PEtALSCDKException("Cannot find properties at location : "
                        + propertiesFilePath);
            }
        }

        return properties;
    }

    /**
     * <p>
     * Replace place holders in the toProperties {@link Properties} object by
     * values found in external properties file denoted by the given
     * propertiesFilePath. Place holders look like :
     * ${property.name.in.fromProperties.object}
     * </p>
     * <p>
     * Exemple : <br/>
     * toProperties property place holder : ${foo.bar}<br/>
     * External properties file property : foo.bar <br/>
     * Property with value ${foo.bar} in toProperties object will be overiden by
     * the value of foo.bar property of the external properties file
     * </p>
     * 
     * @param propertiesFilePath
     *            The file path can be relative path based on Petals root path
     *            or a valid URL
     *            <p>
     *            Exemples : <br/>
     *            Relative path : /conf/external.properties <br/>
     *            Absolute path : file:///C:/conf/external.properties <br/>
     *            URL : ftp://user:pwd@host:port/external.properties
     *            </p>
     * @param toProperties
     *            a {@link Map} instance with some place holder as value to
     *            replace
     * @throws PEtALSCDKException
     *             if error occurs when reading external properties file or if a
     *             property specified in toProperties cannot be found in
     *             external properties file
     */
    public static void fillPlaceHolder(final String propertiesFilePath, final Map<String, String> extensions)
    throws PEtALSCDKException {
        final Properties externalProperties = loadProperties(propertiesFilePath);
        fillPlaceHolder(externalProperties, extensions);
    }

    /**
     * <p>
     * Replace place holders in the toProperties {@link Properties} object by
     * values found in fromProperties {@link Properties} object. Place holders
     * look like : ${property name in fromProperties object}
     * </p>
     * <p>
     * Exemple : <br/>
     * toProperties property place holder : ${foo.bar}<br/>
     * fromProperties property : foo.bar <br/>
     * Property with value ${foo.bar} in toProperties object will be overiden by
     * the value of foo.bar property of the fromProperties object
     * </p>
     * 
     * @param fromProperties
     *            properties that will replace place holders in toProperties
     *            object
     * @param toProperties
     *            a {@link Map} instance with some place holder as value to
     *            replace
     * @throws PEtALSCDKException
     *             if a property specified in toProperties cannot be found in
     *             fromProperties
     */
    public static void fillPlaceHolder(final Properties fromProperties,
            final Map<String, String> extensions) {

        for (final Entry<String, String> entry : extensions.entrySet()) {
            final String entryValue = entry.getValue();
            final String key = entry.getKey();
            if (entryValue != null && entryValue.indexOf("${") >= 0) {
                final List<String> placeHolderKeys = extractPlaceHolderKeys(entryValue);
                final String newEntryValue = fillPlaceHolder(entryValue, placeHolderKeys, fromProperties);
                extensions.put(key, newEntryValue);
            }
        }

    }

    public static List<String> extractPlaceHolderKeys(final String entryValue) {
        final List<String> result = new ArrayList<String>();
        final String[] keys = entryValue.split("[${]");
        for (final String string : keys) {
            if (string.indexOf("}") >= 0) {
                result.add(string.split("[}]")[0]);
            }
        }
        return result;
    }

    public static String fillPlaceHolder(final String entryValue,
            final List<String> placeHolderKeys, final Properties fromProperties) {
        String result = entryValue;
        for (final String key : placeHolderKeys) {
            final String extPropertyValue = fromProperties.getProperty(key);
            if (extPropertyValue != null) {
                result = result.replace("${" + key + "}", extPropertyValue);
            } else {
                result = result.replace("${" + key + "}", "");
            }
        }
        return result;
    }

    /**
     * 
     * @param propertiesFilePath
     * @return
     */
    private static final InputStream loadPropertiesFileFromRelativePath(final String propertiesFilePath) {
        final InputStream result = null;
        // FIXME : really PEtALS dependent..
        final String petalsHome = System.getenv("PETALS_HOME");
        if (petalsHome != null) {
            final File propFile = new File(petalsHome, propertiesFilePath);
            if (propFile.exists()) {
                try {
                    return new FileInputStream(propFile);
                } catch (final FileNotFoundException e) {
                    // TODO log cannot read specified properties file :
                    // propertiesFilePath;
                }
            }
        }
        return result;
    }

    private static final InputStream loadPropertiesFileFromUrl(final URL url) {
        InputStream result = null;
        try {
            result = getInputStream(url);
        } catch (final Exception e) {
            // TODO log can not retrieve properties file from URL :" + url);
        }
        return result;
    }

    private static final InputStream getInputStream(final URL url) throws PEtALSCDKException {
        URLConnection connection = null;
        InputStream is = null;

        try {
            connection = url.openConnection();
            is = connection.getInputStream();
        } catch (final IOException e) {
            throw new PEtALSCDKException("Can not get InputStream from URL " + url, e);
        }
        return is;
    }
}
