/**
 * PETALS - PETALS Services Platform. Copyright (c) 2005 EBM Websourcing,
 * http://www.ebmwebsourcing.com/
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version. This library is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 * ---------------------------------------------------------------------------
 * ---------------------------------------------------------------------------
 */
package org.ow2.petals.component.framework.api.util;

import javax.xml.namespace.QName;
import javax.xml.transform.Source;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamSource;

import org.ow2.petals.component.framework.api.exception.DocumentException;
import org.ow2.petals.component.framework.api.exception.FaultException;
import org.ow2.petals.component.framework.api.exception.PEtALSCDKException;
import org.w3c.dom.Document;

/**
 * Contains utilities for the Source management.
 * 
 * @author Frederic Gardes
 */
public interface SourceUtil {

    /**
     * <p>
     * Creates an SOAP fault, as a DOM document, according to provided exception
     * and code:<br>
     * <code>&lt;s:Fault xmlns:s="http://schemas.xmlsoap.org/soap/envelope/"&gt;<br>
     * &nbsp;&nbsp;&nbsp;&lt;faultcode&gt;{org.objectweb.petals}ERROR&lt;/faultcode&gt;<br>
     * &nbsp;&nbsp;&nbsp;&lt;faultstring&gt;mesage&lt;/faultstring&gt;<br>
     * &nbsp;&nbsp;&nbsp;&lt;faultactor&gt;component&lt;/faultactor&gt;<br>
     * &nbsp;&nbsp;&nbsp;&lt;detail&gt;<br>
     * &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;...<br>
     * &nbsp;&nbsp;&nbsp;&lt;/detail&gt;<br>
     * &lt;s:Fault&gt;</code>
     * </p>
     * 
     * <p>
     * If the provided exception is an instance of {@link DocumentException}:
     * <ul>
     * <li>the fault code is set with the value returned by
     * {@link DocumentException#getFaultCode()},</li>
     * <li>the fault string is set with the value returned by
     * {@link DocumentException#getMessage()},</li>
     * <li>the fault actor is set with the value returned by
     * {@link DocumentException#getFaultActor()},</li>
     * <li>the detail is set with the exception document returned by
     * {@link DocumentException#getDocument()}.</li>
     * </ul>
     * </p>
     * 
     * <p>
     * If the provided exception is an instance of {@link FaultException}:
     * <ul>
     * <li>the fault code is set with the value returned by
     * {@link FaultException#getFaultCode()},</li>
     * <li>the fault string is set with the value returned by
     * {@link FaultException#getMessage()},</li>
     * <li>the fault actor is set with the value returned by
     * {@link FaultException#getFaultActor()},</li>
     * <li>the detail is set with the exception stack trace enclosed in the node
     * <code>&lt;stackTrace&gt;</code>.</li>
     * </ul>
     * </p>
     * 
     * <p>
     * otherwise:
     * <ul>
     * <li>the fault code is set with the value of <code>errorCode</code>,</li>
     * <li>the fault string is set with the value returned by
     * {@link Exception#getMessage()},</li>
     * <li>the fault actor is set to <code>null</code>,</li>
     * <li>the detail is set with the exception stack trace enclosed in the node
     * <code>&lt;stackTrace&gt;</code>.</li>
     * </ul>
     * </p>
     * 
     * @param e
     *            An exception used to provide faultstring and detail
     * @param errorCode
     *            The QName of the fault code. It is not used if the exception
     *            is an instance of {@link FaultException} or
     *            {@link DocumentException}.
     * @return A Document containing the Soap Fault
     */
    Document createSoapFaultDocument(final Throwable e, final QName errorCode);

    /**
     * Create a Source from the String. The message must be a valid XML String.
     * If an encoding is specified in the XML String header it is used to create
     * the XML bytes stream, otherwise the default JVM encoding is used.
     * 
     * @param message
     *            a valid XML String
     * @return the Source built from the XML String
     * @throws PEtALSCDKException
     *             if an error occurs during Source creation
     * 
     */
    Source createSource(final String message) throws PEtALSCDKException;

    /**
     * Create a Source from the String with a specified encoding format. The
     * message must be a valid XML String. The encoding must match the XML
     * declaration in the message.
     * 
     * @param message
     *            a valid XML String
     * @param charFormat
     *            the encoding format, must be non {@code null}
     * 
     * @return the Source built from the XML String
     * @throws PEtALSCDKException
     *             if an error occurs during Source creation
     */
    Source createSource(final String message, final String charFormat) throws PEtALSCDKException;

    /**
     * Create a DOM Source from the DOM document.
     * 
     * @param document
     *            A DOM document
     * @return the DOM source built from the DOM document
     */
    DOMSource createDOMSource(final Document document);

    /**
     * Create a Stream Source from the DOM document.
     * 
     * @param document
     *            A DOM document
     * @return the Stream source built from the DOM document
     */
    StreamSource createStreamSource(final Document document) throws PEtALSCDKException;

    /**
     * Create a {@link Document} from a {@link Source}. <br>
     * -If the Source is a {@link StreamSource}, the source would be still
     * available.<br>
     * -If the Source is a {@link DomSource} and contains a {@link Document},
     * the returned document is a clone of this document.
     * 
     * @param source
     *            the {@link Source}
     * @return the DOM document built from the Source
     * @throws PEtALSCDKException
     */
    Document createDocument(final Source source) throws PEtALSCDKException;

    /**
     * Create a {@link Document} from a {@link Source}. <br>
     * -If the Source is a {@link StreamSource} and {@code forkSource} is set to
     * {@code true}, the source would be still available. Otherwise the source
     * is no more available.<br>
     * If the Source is a {@link DomSource} and contains a {@link Document} and
     * {@code forkSource} is set to {@code true}, the returned document is a
     * clone of this document. Otherwise the original {@link Document} is
     * returned.
     * 
     * @param source
     *            the {@link Source}
     * @param forkSource
     *            {@code true} to fork the content of the source if necessary
     * @return a DOM document
     * @throws PEtALSCDKException
     */
    Document createDocument(final Source source, final boolean forkSource)
    throws PEtALSCDKException;

    /**
     * Create a {@link String} from a {@link Source}.<br>
     * -If the Source is a {@code StreamSource}, the source would be still
     * available.<br>
     * 
     * @param source
     *            the {@link Source}
     * @return the String built from the Source
     * @throws PEtALSCDKException
     */
    String createString(final Source source) throws PEtALSCDKException;

    /**
     * Create a {@link String} from a {@link Source}.<br>
     * -If the Source is a {@link StreamSource} and {@code forkSource} is set to
     * {@code true}, the source would be still available. Otherwise the source
     * is no more available.<br>
     * 
     * @param source
     *            the {@link Source}
     * @return the String built from the Source
     * @throws PEtALSCDKException
     */
    String createString(final Source source, final boolean forkSource) throws PEtALSCDKException;

    /**
     * Test if the document is a DOM3 document
     * 
     * @param document
     *            the document to test
     * @return true if the document is DOM3, false else
     */
    boolean isDOM3(Document document);
}
