/**
 * PETALS - PETALS Services Platform. Copyright (c) 2005 EBM Websourcing,
 * http://www.ebmwebsourcing.com/
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version. This library is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 * ---------------------------------------------------------------------------
 * ---------------------------------------------------------------------------
 */
package org.ow2.petals.component.framework.api.util;

import java.io.File;
import java.net.URL;
import java.util.List;

import javax.jbi.servicedesc.ServiceEndpoint;
import javax.xml.namespace.QName;

import org.ow2.easywsdl.extensions.wsdl4complexwsdl.api.Description;
import org.ow2.easywsdl.wsdl.api.Endpoint;
import org.ow2.easywsdl.wsdl.api.WSDLException;
import org.ow2.petals.component.framework.api.Constants;
import org.w3c.dom.Document;

/**
 * Contains utilities for the WSDL management.
 * 
 * @author Frederic Gardes
 */
public interface WSDLUtil {

    /**
     * Convert the description to a document
     * 
     * @param wsdlDescription
     *            the description
     * @return the document
     * @throws WSDLException
     */
    Document convertDescriptionToDocument(Description wsdlDescription) throws WSDLException;

    /**
     * Write a description into a file.
     * 
     * @param description
     *            the wsdl description
     * @param wsdlFile
     *            The location of the destination wsdl file.
     * @throws WSDLException
     */
    void convertDescriptionToFile(final Description description, final File wsdlFile)
            throws WSDLException;

    /**
     * Convert the description to a string
     * 
     * @param wsdlDescription
     *            the description
     * @return the string
     * @throws WSDLException
     */
    String convertDescriptionToString(Description wsdlDescription) throws WSDLException;

    /**
     * COnvert a document to a description
     * 
     * @param document
     * @return
     * @throws WSDLException
     */
    Description convertDocumentToDescription(Document document) throws WSDLException;

    /**
     * Create a light WSDL2.0 description from the given ServiceEndpoint. All
     * attributes are required into the function parameter, otherwise a
     * NullPointerException will be thrown. Namespaces are managed.
     * 
     * @param interfaceQName
     *            must be non null
     * @param serviceQName
     *            must be non null
     * @param endpointName
     *            must be non null
     * @return a WSDL2.0 light description
     * @throws WSDLException
     * @see 
     *      org.ow2.petals.component.framework.util#createLightWSDL20Description(
     *      final QName[], final QName, final String)
     */
    Description createLightWSDL20Description(final QName interfaceQName, final QName serviceQName,
            final String endpointName) throws WSDLException;

    /**
     * Create a WSDL2.0 enpoint from the given ServiceEndpoint. All attributes
     * are required into the function parameter, otherwise a
     * NullPointerException will be thrown. Namespaces are managed.
     * 
     * @param serviceEndpoint
     *            must be non null and must contain non-null attributes
     * @return a WSDL 2.0 endpoint
     * @throws WSDLException
     * @see 
     *      org.ow2.petals.component.framework.util#createLightWSDL20Description(
     *      final QName[], final QName, final String)
     */
    Endpoint createWSDL20Endpoint(ServiceEndpoint serviceEndpoint) throws WSDLException;

    /**
     * Create a wsdl description. <br/>
     * NOTE : Imported document are written in the description.
     * 
     * @param wsdlLocation
     *            the location of the wsdl file
     * @return the description of the document
     * @throws WSDLException
     */
    Description createWsdlDescription(final URL wsdlLocation) throws WSDLException;

    /**
     * Create a wsdl description. <br/>
     * NOTE : Imported document are written in the description.
     * 
     * @param wsdlLocation
     *            the location of the wsdl file
     * @param processAutogenerateEndpoint
     *            true if you want to replace the endpoints named
     *            {@link Constants.Component#AUTOGENERATED_ENDPOINT_NAME}<br />
     *            by a name generated by PEtALS
     * @return the description of the document
     * @throws WSDLException
     */
    Description createWsdlDescription(URL wsdlLocation, boolean processAutogenerateEndpoint)
            throws WSDLException;

    /**
     * Create a wsdl description.
     * 
     * @param wsdlLocation
     *            the location of the wsdl file
     * @param processAutogenerateEndpoint
     *            true if you want to replace the endpoints named
     *            {@link Constants.Component#AUTOGENERATED_ENDPOINT_NAME}<br />
     *            by a name generated by PEtALS
     * @param addImportedDocumentsInWsdl
     *            if true the imported document are written in the description.
     * @return the description of the document
     * @throws WSDLException
     */
    Description createWsdlDescription(final URL wsdlLocation,
            final boolean processAutogenerateEndpoint, final boolean addImportedDocumentsInWsdl)
            throws WSDLException;

    /**
     * Get the endpoints into a wsdl description.
     * 
     * @param wsdlDescription
     *            the wsdl description generated by easy wsdl
     * @return The list of Endpoint, or null if the description is null
     */
    List<Endpoint> getEndpointList(Description wsdlDescription);

    /**
     * Get the endpoints into a wsdl description. Retrieve all of them if no
     * filter name is provided.<br />
     * Retrieve only the endpoints matching the service QName filter if
     * provided.<br />
     * Retrieve only the endpoints matching the endpoint name filter if
     * provided.
     * 
     * @param wsdlDescription
     *            the wsdl description generated by easy wsdl
     * @param serviceQNameFilter
     *            if not null, only the endpoints with this service QName will
     *            be return
     * @param endpointNameFilter
     *            if not null, only the endpoints with this name will be return
     * @return The list of endpoints, empty if nothing found
     */
    abstract List<Endpoint> getEndpointList(Description wsdlDescription, QName serviceQNameFilter,
            String endpointNameFilter);

    /**
     * Check if a description contains a specific endpoint
     * 
     * @param wsdlDescription
     *            the description
     * @param endpoint
     *            the endpoint. must be non null, with a service non null and a
     *            name non null
     * @return true if the description contains the endpoint, false else
     */
    boolean isDescriptionContaining(Description wsdlDescription, ServiceEndpoint endpoint);
}
