/**
 * PETALS - PETALS Services Platform. Copyright (c) 2005 EBM Websourcing,
 * http://www.ebmwebsourcing.com/
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version. This library is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 * ---------------------------------------------------------------------------
 * ---------------------------------------------------------------------------
 */
package org.ow2.petals.component.framework.api.util;

import java.io.IOException;
import java.util.Hashtable;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactoryConfigurationError;

import org.w3c.dom.Document;
import org.xml.sax.SAXException;

/**
 * 
 * 
 * @author Frederic Gardes
 */
public interface XSLTUtil {
    /**
     * Loads a W3C XML document from a file.
     * 
     * @param filename
     *            The name of the file to be loaded
     * @return a document object model object representing the XML file
     * @throws IOException
     * @throws ParserConfigurationException
     * @throws SAXException
     */
    public Document loadXML(String filename) throws IOException, ParserConfigurationException,
    SAXException;

    /**
     * Applies a stylesheet to a given xml document.
     * 
     * @param xmlDocument
     *            the xml document to be transformed
     * @param xsltFilename
     *            the filename of the stylesheet
     * @return the transformed xml document
     * @throws ParserConfigurationException
     * @throws TransformerException
     */
    public Document transformDocument(Document xmlDocument, String xsltFilename)
    throws TransformerException, ParserConfigurationException;

    /**
     * Applies a stylesheet (that receives parameters) to a given xml document.
     * 
     * @param xmlDocument
     *            the xml document to be transformed
     * @param parameters
     *            the hashtable with the parameters to be passed to the
     *            stylesheet
     * @param xsltFilename
     *            the filename of the stylesheet
     * @return the transformed xml document
     * @throws ParserConfigurationException
     * @throws TransformerException
     */
    public Document transformDocument(Document xmlDocument, Hashtable<String, String> parameters,
            String xsltFilename) throws TransformerException, ParserConfigurationException;

    /**
     * Applies a stylesheet (that receives parameters) to a given xml document.
     * The resulting XML document is converted to a string after transformation.
     * The order of the attributes of an element can't change between the
     * stylesheet and the string generated.
     * 
     * @param xmlDocument
     *            the xml document to be transformed
     * @param parameters
     *            the hashtable with the parameters to be passed to the
     *            stylesheet
     * @param xsltFilename
     *            the filename of the stylesheet
     * @return the transformed xml document as a string
     * @throws TransformerException
     * @throws TransformerFactoryConfigurationError
     */
    public String transformDocumentAsString(Document xmlDocument,
            Hashtable<String, String> parameters, String xsltFilename)
    throws TransformerFactoryConfigurationError, TransformerException;

    /**
     * Applies a stylesheet to a given xml document. The order of the attributes
     * of an element can't change between the stylesheet and the string
     * generated.
     * 
     * @param xmlDocument
     *            the xml document to be transformed
     * @param xsltFilename
     *            the filename of the stylesheet
     * @return the transformed xml document
     * @throws TransformerException
     * @throws TransformerFactoryConfigurationError
     */
    public String transformDocumentAsString(Document xmlDocument, String xsltFilename)
    throws TransformerFactoryConfigurationError, TransformerException;

    /**
     * Applies a stylesheet to a given xml document. The order of the attributes
     * of an element can't change between the stylesheet and the string
     * generated.
     * 
     * @param xmlDocument
     *            the xml document to be transformed
     * @param xsltSource
     *            the source of the stylesheet
     * @return the transformed xml document
     * @throws TransformerException
     * @throws TransformerFactoryConfigurationError
     */
    public String transformDocumentAsString(Document xmlDocument, Source xsltSource)
    throws TransformerFactoryConfigurationError, TransformerException;

    /**
     * Applies a stylesheet (that receives parameters) to a given xml document.
     * The resulting XML document is converted to a string after transformation.
     * The order of the attributes of an element can't change between the
     * stylesheet and the string generated.
     * 
     * @param xmlDocument
     *            the xml document to be transformed
     * @param parameters
     *            the hashtable with the parameters to be passed to the
     *            stylesheet
     * @param xsltSource
     *            the source of the stylesheet
     * @return the transformed xml document as a string
     * @throws TransformerException
     * @throws TransformerFactoryConfigurationError
     */
    public String transformDocumentAsString(Document xmlDocument,
            Hashtable<String, String> parameters, Source xsltSource)
            throws TransformerFactoryConfigurationError, TransformerException;

    /**
     * Applies a stylesheet to a given xml document.
     * 
     * @param xmlDocument
     *            the xml document to be transformed
     * @param xsltSource
     *            the source of the stylesheet
     * @return the transformed xml document
     */
    public Document transformDocument(Document xmlDocument, Source xsltSource)
    throws TransformerException, ParserConfigurationException;

    /**
     * Applies a stylesheet (that receives parameters) to a given xml document.
     * 
     * @param xmlDocument
     *            the xml document to be transformed
     * @param parameters
     *            the hashtable with the parameters to be passed to the
     *            stylesheet
     * @param xsltSource
     *            the source of the stylesheet
     * @return the transformed xml document
     */
    public Document transformDocument(Document xmlDocument, Hashtable<String, String> parameters,
            Source xsltSource) throws TransformerException, ParserConfigurationException;

    /**
     * Use a transformer for a given xml document.
     * 
     * @param xmlDocument
     *            the xml document to be transformed
     * @param transformer
     *            the transformer created with the stylesheet
     * 
     * @return the transformed xml document
     */
    public Document transformDocument(final Document xmlDocument, final Transformer transformer)
    throws TransformerException, ParserConfigurationException;

    /**
     * Create the transformer
     * 
     * @param parameters
     *            the hashtable with the parameters to be passed to the
     *            stylesheet
     * @param xsltSource
     *            the source of the stylesheet
     * @return the transformer
     * @throws TransformerFactoryConfigurationError
     * @throws TransformerException
     */
    public Transformer createTransformer(final Hashtable<String, String> parameters,
            final Source xsltSource) throws TransformerFactoryConfigurationError,
            TransformerException;

    /**
     * Create a transformer
     * 
     * @param xsltSource
     *            the source of the stylesheet
     * @return the transformer
     * @throws TransformerFactoryConfigurationError
     * @throws TransformerException
     */
    public Transformer createTransformer(final Source xsltSource)
    throws TransformerFactoryConfigurationError, TransformerException;

}
