/**
 * Copyright (c) 2012-2013 Linagora
 * 
 * This program/library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This program/library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program/library; If not, see <http://www.gnu.org/licenses/>
 * for the GNU Lesser General Public License version 2.1.
 */
package org.ow2.petals.cli.extension.command.monitoring.mo.api;

import java.util.Iterator;
import java.util.ServiceLoader;
import java.util.logging.Logger;

import org.ow2.petals.cli.extension.command.monitoring.mo.api.exception.DuplicatedMonitoringObjectException;
import org.ow2.petals.cli.extension.command.monitoring.mo.api.exception.DuplicatedMonitoringSubFunctionException;
import org.ow2.petals.cli.extension.command.monitoring.mo.api.exception.InvalidMonitoringObjectNameException;
import org.ow2.petals.cli.extension.command.monitoring.mo.api.exception.InvalidMonitoringSubFunctionNameException;

/**
 * Monitoring object registrator introspecting the class loader to find
 * available monitoring objects.
 * 
 * @author Christophe DENEUX - Linagora
 * 
 */
public class MonitoringObjectRegistration {

    private final static Logger LOG = Logger
            .getLogger(MonitoringObjectRegistration.class.getName());

    public final static void registers(final MonitoringObjectRegistror monitoringObjectRegistror) {
        final ServiceLoader<MonitoringObject> monitoringObjectsLoader = ServiceLoader
                .load(MonitoringObject.class);
        final Iterator<MonitoringObject> monitoringObjectsIterator = monitoringObjectsLoader
                .iterator();
        while (monitoringObjectsIterator.hasNext()) {
            try {
                final MonitoringObject monitoringObject = monitoringObjectsIterator.next();
                monitoringObjectRegistror.registersMonitoringObjects(monitoringObject);
                LOG.fine("Monitoring object '" + monitoringObject.getName() + "' registered.");
            } catch (final DuplicatedMonitoringObjectException e) {
                // Monitoring object already registered
                LOG.severe("Trying to register a monitoring object already registered ('"
                        + e.getMonitoringObject().getName() + "'). Ignored.");
            } catch (final DuplicatedMonitoringSubFunctionException e) {
                // Monitoring sub-function registered twice
                LOG.severe("Trying to register a monitoring sub-function ('"
                        + e.getMonitoringSubFunction().getName()
                        + "') twice into a monitoring object ('"
                        + e.getMonitoringObject().getName() + "'). Ignored.");
            } catch (final InvalidMonitoringObjectNameException e) {
                // Monitoring object name is invalid
                LOG.severe("Trying to register a monitoring object having an invalid name ('"
                        + e.getMonitoringObject().getName() + "'). Ignored.");
            } catch (final InvalidMonitoringSubFunctionNameException e) {
                // Monitoring sub-function name is invalid
                LOG.severe("Trying to register a monitoring object having a sub-function with an invalid name ('"
                        + e.getMonitoringObject().getName()
                        + "', '"
                        + e.getMonitoringSubFunction().getName() + "'). Ignored.");
            } catch (final IllegalArgumentException e) {
                // It's a bug
                LOG.severe("Trying to register a null monitoring object. Ignored.");
            }
        }
    }

}
