/**
 * Copyright (c) 2012-2013 Linagora
 * 
 * This program/library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This program/library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program/library; If not, see <http://www.gnu.org/licenses/>
 * for the GNU Lesser General Public License version 2.1.
 */
package org.ow2.petals.cli.extension.command.monitoring.mo.api;

import static org.junit.Assert.assertEquals;

import java.io.IOException;
import java.net.URL;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;

import org.junit.Test;
import org.ow2.petals.cli.api.shell.Shell;
import org.ow2.petals.cli.extension.command.monitoring.mo.api.exception.DuplicatedMonitoringObjectException;
import org.ow2.petals.cli.extension.command.monitoring.mo.api.exception.InvalidMonitoringObjectNameException;

public class MonitoringObjectRegistrationTestCase {

    private class MonitoringObjectRegistrorTest implements MonitoringObjectRegistror {

        private final Map<String, MonitoringObject> monitoringObjects = new HashMap<String, MonitoringObject>();

        @Override
        public void registersMonitoringObjects(final MonitoringObject monitoringObject)
                throws DuplicatedMonitoringObjectException, InvalidMonitoringObjectNameException,
                IllegalArgumentException {
            if (monitoringObject.getName() == null) {
                throw new InvalidMonitoringObjectNameException(monitoringObject);
            } else if (monitoringObject.getName().equals("duplicated")) {
                throw new DuplicatedMonitoringObjectException(monitoringObject);
            } else {
                this.monitoringObjects.put(monitoringObject.getName(), monitoringObject);
            }
        }

        @Override
        public Map<String, MonitoringObject> getMonitoringObjects() {
            return this.monitoringObjects;
        }

        @Override
        public Shell getShell() {
            return null;
        }

    };

    /**
     * Check the automatic registration of monitoring objects
     */
    @Test
    public void testRegistersMonitoringObjects_00() {

        final MonitoringObjectRegistror monitoringObjRegistrator = new MonitoringObjectRegistrorTest();
        MonitoringObjectRegistration.registers(monitoringObjRegistrator);
        final MonitoringObject dummyMonitoringObject = new DummyMonitoringObject();

        final Map<String, MonitoringObject> monitoringObjects = monitoringObjRegistrator
                .getMonitoringObjects();
        assertEquals("Too much monitoring objects registered", 1, monitoringObjects.size());
        assertEquals("Unexpected registered monitoring object", dummyMonitoringObject.getName(),
                monitoringObjects.get(dummyMonitoringObject.getName()).getName());
    }

    /**
     * Check the registration of monitoring object having no name (name =
     * <code>null</code>)
     */
    @Test
    public void testRegistersMonitoringObjects_01() {

        final ClassLoader oldClassloader = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(new ClassLoader(oldClassloader) {

            @Override
            public Enumeration<URL> getResources(final String name) throws IOException {
                if (name.equals("META-INF/services/" + MonitoringObject.class.getName())) {
                    return super.getResources("spi-null");
                } else {
                    return super.getResources(name);
                }
            }
        });

        try {
            final MonitoringObjectRegistror monitoringObjRegistrator = new MonitoringObjectRegistrorTest();
            MonitoringObjectRegistration.registers(monitoringObjRegistrator);
            final Map<String, MonitoringObject> monitoringObjects = monitoringObjRegistrator
                    .getMonitoringObjects();
            assertEquals("Too much monitoring objects registered", 0, monitoringObjects.size());
        } finally {
            Thread.currentThread().setContextClassLoader(oldClassloader);
        }
    }

    /**
     * Check the registration of a monitoring object already registered
     */
    @Test
    public void testRegistersMonitoringObjects_02() {

        final ClassLoader oldClassloader = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(new ClassLoader(oldClassloader) {

            @Override
            public Enumeration<URL> getResources(final String name) throws IOException {
                if (name.equals("META-INF/services/" + MonitoringObject.class.getName())) {
                    return super.getResources("spi-duplicated");
                } else {
                    return super.getResources(name);
                }
            }
        });

        try {
            final MonitoringObjectRegistror monitoringObjRegistrator = new MonitoringObjectRegistrorTest();
            MonitoringObjectRegistration.registers(monitoringObjRegistrator);
            final Map<String, MonitoringObject> monitoringObjects = monitoringObjRegistrator
                    .getMonitoringObjects();

            final MonitoringObject dummyMonitoringObject = new DummyMonitoringObject();
            assertEquals("Too much monitoring objects registered", 1, monitoringObjects.size());
            assertEquals("Unexpected registered monitoring object",
                    dummyMonitoringObject.getName(),
                    monitoringObjects.get(dummyMonitoringObject.getName()).getName());
        } finally {
            Thread.currentThread().setContextClassLoader(oldClassloader);
        }
    }
}
