/**
 * PETALS - PETALS Services Platform.
 * Copyright (c) 2007 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * TranscoderSUListener.java 
 * -------------------------------------------------------------------------
 */

package org.ow2.petals.se.transcoder;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.ow2.petals.component.framework.exception.PEtALSCDKException;
import org.ow2.petals.component.framework.su.ServiceUnitListener;
import org.ow2.petals.component.framework.util.Extensions;
import org.ow2.petals.jbi.descriptor.JBIDescriptor;
import org.ow2.petals.jbi.descriptor.Provides;

/**
 * 
 * @author Anne-Marie BARTHE (ambarthe) - eBMWebSourcing
 * 
 */
public class TranscoderSUListener implements ServiceUnitListener {

    private Map<String, Connection> connections;

    /**
     * Default constructor
     */
    public TranscoderSUListener() {
        super();
        this.connections = new HashMap<String, Connection>();
    }

    /**
     * This is used to get a connection with the HSQLBD database
     * 
     * @param driverClass
     *            the JDBC driver
     * @param url
     *            the URL of the database
     * @param username
     *            the user name
     * @param password
     *            the password
     * @return
     * @throws TranscoderException
     */
    private Connection getConnection(String driverClass, String url, String username,
            String password) throws TranscoderException {
        try {
            Class.forName(driverClass).newInstance();
            final Connection connection = DriverManager.getConnection(url, username, password);
            return connection;
        } catch (final Exception e) {
            throw new TranscoderException("Can't get a connection : ", e);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.ow2.petals.component.framework.su.ServiceUnitListener#onSUDeployed(java.lang.String,
     *      java.lang.String, org.ow2.petals.jbi.descriptor.JBIDescriptor)
     */
    public void onSUDeployed(final String serviceUnitName, final String suRootPath,
            final JBIDescriptor jbiDescriptor) throws PEtALSCDKException {
        Connection connection = null;

        // Retrieve all the provides for each jbiDescriptor
        final List<Provides> providesList = jbiDescriptor.getServices().getProvides();

        for (final Provides provides : providesList) {
            // Retrieve all extensions for each provides
            final Extensions extensions = new Extensions(provides.getExtensions());
            if (extensions != null) {
                final Properties props = extensions.getPetalsExtensions();
                // Get parameters from extensions
                final String driverClass = props.getProperty(Constants.DRIVER_CLASS);
                final String url = props.getProperty(Constants.URL_DB);
                final String username = props.getProperty(Constants.USRNAME_DB);
                final String password = props.getProperty(Constants.PWD_DB);
                // Create the connection
                connection = this.getConnection(driverClass, url, username, password);
                // Store the connections
                this.connections.put(provides.getEndpointName(), connection);
            }
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.ow2.petals.component.framework.su.ServiceUnitListener#onSUStarted(java.lang.String)
     */
    public void onSUStarted(String arg0) throws PEtALSCDKException {
        // Do nothing
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.ow2.petals.component.framework.su.ServiceUnitListener#onSUStopped(java.lang.String)
     */
    public void onSUStopped(String arg0) throws PEtALSCDKException {
        // Do nothing
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.ow2.petals.component.framework.su.ServiceUnitListener#onSUUndeployed(java.lang.String,
     *      java.lang.String, org.ow2.petals.jbi.descriptor.JBIDescriptor)
     */
    public void onSUUndeployed(final String serviceUnitName, final String jbiSuRootPath,
            final JBIDescriptor jbiDescriptor) throws PEtALSCDKException {
        Connection connection = null;

        // Retrieve all the provides for each jbiDescriptor
        final List<Provides> providesList = jbiDescriptor.getServices().getProvides();

        for (final Provides provides : providesList) {
            // Retrieve all extensions for each provides
            final Extensions extensions = new Extensions(provides.getExtensions());
            if (extensions != null) {
                // Remove the connection linked to the current endpoint
                connection = this.connections.remove(provides.getEndpointName());
                try {
                    connection.close();
                } catch (final SQLException e) {
                    // FIXME : Not sure that we must throw such exception on
                    // connection close
                    throw new PEtALSCDKException("Can not close the DB connection : ", e);
                }
            }
        }
    }

    public Map<String, Connection> getConnections() {
        return this.connections;
    }
}
