/**
 * PETALS - PETALS Services Platform.
 * Copyright (c) 2007 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * JBIListener.java
 * -------------------------------------------------------------------------
 */

package org.ow2.petals.se.transcoder.listener;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;

import javax.jbi.messaging.MessagingException;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.ow2.petals.component.framework.listener.AbstractJBIListener;
import org.ow2.petals.component.framework.util.Exchange;
import org.ow2.petals.se.transcoder.Constants;
import org.ow2.petals.se.transcoder.Transcoder;
import org.ow2.petals.se.transcoder.TranscoderEngine;
import org.ow2.petals.se.transcoder.TranscoderImpl;
import org.ow2.petals.se.transcoder.TranscoderSUListener;
import org.ow2.petals.se.transcoder.util.TranscoderQuery;
import org.ow2.petals.se.transcoder.util.XMLTagContent;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

/**
 * The Transcoder JBIListener
 * 
 * @author Anne-Marie BARTHE (ambarthe) - EBM Websourcing
 * 
 */
public class JBIListener extends AbstractJBIListener {

    private Transcoder transcoder;

    private TranscoderSUListener suListener;

    private Connection connection;

    /**
     * Constructor
     */
    public JBIListener() {
        super();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.ow2.petals.component.framework.listener.AbstractListener#init()
     */
    @Override
    protected void init() {
        this.transcoder = new TranscoderImpl();
        // Set service unit listener
        this.suListener = (TranscoderSUListener) ((TranscoderEngine) this.getComponent())
                .getServiceUnitListener();

    }

    /*
     * (non-Javadoc)
     * 
     * @see org.ow2.petals.component.framework.listener.AbstractJBIListener#onJBIMessage(org.ow2.petals.component.framework.util.Exchange)
     */
    @Override
    protected void onJBIMessage(Exchange exchange) {
        try {
            final String operation = exchange.getOperationName();

            if (exchange.isInOutPattern() && operation.equals(Constants.TRANSLATE_OPERATION)) {
                this.getLogger().info("Performing '" + operation + "' operation.");
                this.process(exchange);
            } else if (exchange.isInOnlyPattern()
                    && (operation.equals(Constants.CREATE_OPERATION)
                            || operation.equals(Constants.UPDATE_OPERATION) || operation
                            .equals(Constants.DELETE_OPERATION))) {
                this.getLogger().info("Performing '" + operation + "' operation.");
                this.process(exchange);
            } else {
                exchange.setFault(new Exception("Unknown operation or bad pattern: " + operation
                        + "    " + exchange.getPattern()));
            }

        } catch (final Throwable e) {
            this.getLogger().severe("Can not set fault in message : " + e.getMessage());
        }
    }

    /**
     * 
     * @param exchange
     * @throws Exception
     */
    private void process(Exchange exchange) throws Exception {
        Document payload = null;
        TranscoderQuery transcoderQuery = null;
        ResultSet response = null;

        try {
            // Get the current end point
            final String endpointName = exchange.getEndpointName();

            // Get the linked connection
            this.connection = this.suListener.getConnections().get(endpointName).getMetaData()
                    .getConnection();
            Statement statement = this.connection.createStatement();

            // Get the input message (XML String)
            try {
                payload = exchange.getInMessageContentAsDocument();
            } catch (final MessagingException e) {
                throw new MessagingException("Problem in payload setting : ", e);
            }

            // Process the input source to get the query parameters (text
            // content of the XML tags)
            try {
                transcoderQuery = XMLTagContent.getValues(payload);
            } catch (final Exception e) {
                throw new MessagingException("Problem in getting values" + e);
            }

            // Get the operation name
            final String operation = exchange.getOperationName();

            // Loop : "what is the chosen operation ?"
            if (Constants.TRANSLATE_OPERATION.equalsIgnoreCase(operation)) {
                response = this.transcoder.translate(this.connection, transcoderQuery);
            } else if (Constants.UPDATE_OPERATION.equalsIgnoreCase(operation)) {
                response = this.transcoder.update(this.connection, transcoderQuery);
            } else if (Constants.DELETE_OPERATION.equalsIgnoreCase(operation)) {
                response = this.transcoder.delete(this.connection, transcoderQuery);
            } else if (Constants.CREATE_OPERATION.equalsIgnoreCase(operation)) {
                response = this.transcoder.create(this.connection, transcoderQuery);
            }

            // Set response as an XML document
            if (response != null) {
                exchange.setOutMessageContent(this.transformResultSetIntoDocument(response,
                        transcoderQuery));

            } else {
                exchange.setFault(new Exception("Can not get result for : '" + operation
                        + "' with the given values"));
            }
        } catch (final Exception e) {
            this.getLogger().info("Can not connect to the database : " + e.getMessage());
        } finally {
            try {
                response.close();
            } catch (SQLException e) {
            }
            try {
                // Commit the changes
                this.connection.commit();
            } catch (SQLException e) {
            }
        }
    }

    /**
     * This is used to translate the ResulSet in a XML Document
     * 
     * @param resultset,
     *            the result set of the query
     * @param query
     *            the query object, where the query parameters are set
     * @return
     * @throws ParserConfigurationException
     * @throws SQLException
     */
    private Document transformResultSetIntoDocument(ResultSet resultset, TranscoderQuery query)
            throws ParserConfigurationException, SQLException {
        // The DOM document object is created
        final Document document = DocumentBuilderFactory.newInstance().newDocumentBuilder()
                .newDocument();
        // The root element of the DOM document, named "result" is created
        final Element root = document.createElement("result");
        document.appendChild(root);

        while (resultset.next()) {
            final Node node = document.createTextNode(resultset.getString(query.getColTName()));
            root.appendChild(node);
        }
        return document;
    }
}
