/**
 * OW2 Util
 * Copyright (C) 2006-2008 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: ComponentRegistry.java 4389 2008-12-15 13:48:57Z alitokmen $
 * --------------------------------------------------------------------------
 */

package org.ow2.util.component;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.ow2.util.component.api.Component;
import org.ow2.util.component.api.ComponentException;
import org.ow2.util.component.api.IComponentRegistry;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Registry that manages components. It allows to get components.
 * @author Florent Benoit
 */
public class ComponentRegistry implements IComponentRegistry {

    /**
     * Logger.
     */
    private Log logger = LogFactory.getLog(ComponentRegistry.class);

    /**
     * Map of components.<br/> Name <--> Implementation of the component
     */
    private Map<String, Component> components = null;

    /**
     * Constructor.
     */
    public ComponentRegistry() {
        // init map
        this.components = new HashMap<String, Component>();
    }

    /**
     * Register a component.
     * @param componentName the name of the component to register
     * @param component the component to register.
     * @throws ComponentException if registering fails.
     */
    public void register(final String componentName, final Component component) throws ComponentException {
        // Existing ?
        if (this.components.containsKey(componentName)) {
            throw new ComponentException("Cannot register the component with the name '" + componentName
                    + "'. There is an existing component with this name.");
        }

        this.logger.debug("Registering component with name {0}.", componentName);
        this.components.put(componentName, component);
    }

    /**
     * Unregister a component.
     * @param componentName the component name to unregister.
     * @throws ComponentException if unregistering fails.
     */
    public void unregister(final String componentName) throws ComponentException {
        // Exist ?
        if (!this.components.containsKey(componentName)) {
            throw new ComponentException("No component with the name '" + componentName
                    + "' found. Component not unregistered");
        }

        this.logger.info("Unregistering component with name {0}.", componentName);
        this.components.remove(componentName);
    }

    /**
     * Unregister a component.
     * @param component the instance of the component to unregister.
     * @throws ComponentException if unregistering fails.
     */
    public void unregister(final Component component) throws ComponentException {
        String name = null;

        // Find component
        Set<String> keys = this.components.keySet();
        for (String key : keys) {
            Component foundComponent = this.components.get(key);
            if (foundComponent.equals(component)) {
                // got it !
                name = key;
                break;
            }
        }
        // found --> unregister.
        if (name != null) {
            unregister(name);
        }
        throw new ComponentException("No component found in the registry with the given component '" + component + "'.");

    }

    /**
     * Allow to get a reference on another component.
     * @param componentName the name of the component
     * @return the component.
     */
    public Component getComponent(final String componentName) {
        return this.components.get(componentName);
    }

    /**
     * @param component EZBComponent instance.
     * @return Returns the component name from the EZBComponent instance.
     */
    public String getComponentName(final Component component) {

        // Iterates over the components to find the component's name
        String match = null;
        for (Iterator<String> i = this.components.keySet().iterator();
            i.hasNext() && (match == null);) {
            String key = i.next();
            Component candidate = this.components.get(key);
            if (component.equals(candidate)) {
                match = key;
                break;
            }
        }
        if (match == null) {
            throw new IllegalStateException("Each component should be registered in the registry. No component found for '"
                    + component + "'.");
        }
        return match;
    }


    /**
     * Get the components that implements the given interface.
     * @param itf the given interface
     * @return an array of components implementing the given interface
     * @param <T> an interface extending EZBComponent.
     */
    @SuppressWarnings("unchecked")
    public <T extends Component> List<T> getComponents(final Class<T> itf) {
        // Check not null
        if (itf == null) {
            throw new IllegalArgumentException("Cannot find component with a null interface");
        }

        // Check interface
        if (!itf.isInterface()) {
            throw new IllegalArgumentException("The given class '" + itf + "' is not an interface");
        }

        // Iterates over the components to find a matching component
        List<T> matchComponents = new ArrayList<T>();
        for (Component component : this.components.values()) {
            // Component is implemeting the given interface ?
            if (Arrays.asList(component.getClass().getInterfaces()).contains(itf)) {
                   matchComponents.add((T) component);
            }
        }
        return matchComponents;
    }
}
