/**
 * OW2 Util
 * Copyright (C) 2006-2008 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: ComponentManager.java 4389 2008-12-15 13:48:57Z alitokmen $
 * --------------------------------------------------------------------------
 */

package org.ow2.util.component;

import java.util.ArrayList;
import java.util.List;

import org.ow2.util.component.api.Component;
import org.ow2.util.component.api.ComponentException;
import org.ow2.util.component.api.IComponentManager;
import org.ow2.util.component.api.IComponentRegistry;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Create and destroy components.
 * @author Florent Benoit
 */
public class ComponentManager implements IComponentManager {

    /**
     * If Component classname ends with "Component", safely remove it.
     */
    private static final String COMPONENT_STR = "Component";

    /**
     * Logger.
     */
    private Log logger = LogFactory.getLog(ComponentManager.class);

    /**
     * Components names that are managed.
     */
    private List<String> componentNames = null;

    /**
     * Components objects. (that were set by configuration).
     */
    private Components components = null;


    /**
     * Link to the registry of components (key=component name/value=EZB component).
     */
    private ComponentRegistry componentRegistry = null;

    /**
     * Build a component manager.
     */
    public ComponentManager() {
        this.componentRegistry = new ComponentRegistry();
        this.componentNames = new ArrayList<String>();
    }

    /**
     * Build a new component manager with the given set of components.
     * @param components the given set of components
     */
    public ComponentManager(final Components components) {
        this();
        setComponents(components);
    }


    /**
     * Gets the set of components.
     * @return the set of components.
     */
    public Components getComponents() {
        return this.components;
    }

    /**
     * Sets the components object.
     * @param components the set of components.
     */
    public void setComponents(final Components components) {
        this.components = components;
    }

    /**
     * Add the given component.
     * @param component the component to register.
     * @throws ComponentException if the component is not added.
     */
    public void addComponent(final Component component) throws ComponentException {
        // Add component
        addComponent(getComponentName(component), component);
    }

    /**
     * Remove the given component.
     * @param component the component to unregister.
     * @throws ComponentException if the component is not removed.
     */
    public void removeComponent(final Component component) throws ComponentException {
        // Remove component
        String componentName = this.componentRegistry.getComponentName(component);
        this.componentNames.remove(componentName);
        this.componentRegistry.unregister(componentName);
    }

    /**
     * Gets the name for a given component.
     * @param component the component instance.
     * @return the name of the component.
     */
    private String getComponentName(final Component component) {
        // get name
        String componentName = component.getClass().getCanonicalName();

        // exist ? (increment counter to get an unique id)
        int index = 2;
        if (componentNames.contains(componentName)) {
            while (componentNames.contains(componentName)) {
                componentName = componentName + (index++);
            }
        }
        return componentName;
    }

    /**
     * Add a component.
     * @param componentName the name of the component to add
     * @param component the component to add.
     * @throws ComponentException if adds fails.
     */
    private void addComponent(final String componentName, final Component component) throws ComponentException {
        // register component
        componentRegistry.register(componentName, component);

        // add to manage list
        componentNames.add(componentName);
    }

    /**
     * Init the components by calling init() method.
     * @throws ComponentException if initialization fails
     */
    public void initComponents() throws ComponentException {

        // Exit soon if there is no components
        if (components == null) {
            return;
        }

        // Register component
        List<Component> componentList = components.getComponents();
        if (componentList != null) {
            for (Component component : componentList) {
                addComponent(component);
            }


            // Call init method if any on each component
            for (String componentName : componentNames) {
                Component component = componentRegistry.getComponent(componentName);
                component.init();
            }

        }
    }

    /**
     * Start the components.
     * @throws ComponentException if starting is failing
     */
    public void startComponents() throws ComponentException {

        StringBuilder sb = new StringBuilder();
        sb.append("[ Component(s) started : ");

        // Call init method if any on each component
        for (String componentName : componentNames) {
            Component component = componentRegistry.getComponent(componentName);
            component.start();

            // append the component name
            String name = component.getClass().getSimpleName();
            // remove "Component" substring if any
            if (name.endsWith(COMPONENT_STR)) {
                name = name.substring(0, name.lastIndexOf(COMPONENT_STR));
            }
            sb.append(name);
            sb.append(" ");
        }

        sb.append("]");
        logger.info(sb.toString());

    }

    /**
     * Stop the components.
     */
    public void stopComponents() {

        // Call stop method if any on each component in the reverse order of the start.
        int size = componentNames.size();
        for (int i = size - 1; i >= 0; i--) {
            String componentName = componentNames.get(i);
            Component component = componentRegistry.getComponent(componentName);
            try {
                component.stop();
            } catch (ComponentException e) {
                logger.error("Cannot stop component with name '" + componentName + "'.", e);
            }
        }
    }

    /**
     * @return the component registry used by this manager.
     */
    public IComponentRegistry getComponentRegistry() {
        return componentRegistry;
    }
}
