/**
 * EasyBeans
 * Copyright (C) 2008 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: ReferenceHelper.java 4667 2009-02-20 14:48:33Z benoitf $
 * --------------------------------------------------------------------------
 */

package org.ow2.util.ee.builder.webserviceref;

import java.io.IOException;
import java.io.Serializable;

import javax.naming.BinaryRefAddr;
import javax.naming.RefAddr;
import javax.naming.Reference;
import javax.naming.StringRefAddr;

import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;
import org.ow2.util.marshalling.Serialization;

/**
 * A ReferenceHelper is a class that helps to insert and extracts values
 * in and out of a JNDI Reference.
 * @author Guillaume Sauthier
 */
public class ReferenceHelper {

    /**
     * Logger.
     */
    private static Log logger = LogFactory.getLog(ReferenceHelper.class);

    /**
     * Wrapped reference.
     */
    private Reference reference;

    /**
     * Creates a helper dedicated to the given Reference.
     * @param reference wrapped Reference
     */
    public ReferenceHelper(final Reference reference) {
        this.reference = reference;
    }

    /**
     * Insert into the wrapped Reference the given key/value String pair.
     * @param key the unique key
     * @param value the String value
     * @return this helper for chained invocation.
     */
    public ReferenceHelper insert(final String key, final String value) {
        // Only insert something if the value is not null
        if (value != null) {
            reference.add(new StringRefAddr(key, value));
            logger.debug("String Reference Insertion ''{0}''->''{1}''", key, value);
        }
        return this;
    }

    /**
     * Extract a value from the wrapped Reference using the given key.
     * @param key the key
     * @return the String value or null if the key was not referring to a
     *         StringRefAddr or if the key was not found in the Reference
     */
    public String extract(final String key) {
        RefAddr addr = reference.get(key);
        if ((addr != null) && (addr instanceof StringRefAddr)) {
            StringRefAddr sra = (StringRefAddr) addr;
            return (String) sra.getContent();
        }
        return null;
    }

    /**
     * Insert into the wrapped Reference the given key/value pair.
     * @param key the unique key
     * @param value the Object value that will be serialized
     * @return this helper for chained invocation.
     */
    public ReferenceHelper insert(final String key, final Serializable value) {
        // Only insert something if the value is not null
        if (value != null) {
            try {
                byte[] bytes = Serialization.storeObject(value);
                reference.add(new BinaryRefAddr(key, bytes));
                logger.debug("Binary Reference Insertion ''{0}''->''{1}''", key, value.toString());
            } catch (IOException e) {
                logger.warn("Unable to serialize ''{0}''", value.toString(), e);
            }
        }
        return this;
    }

    /**
     * Extract a value from the wrapped Reference using the given key.
     * @param key the key
     * @param type the expected classtype to be deserialized
     * @return the deserialized value or null if the key was not referring to a
     *         BinaryRefAddr or if the key was not found in the Reference or if
     *         there was an error during deserialization
     */
    public <T> T extract(final String key, final Class<T> type) {
        RefAddr addr = reference.get(key);
        if ((addr != null) && (addr instanceof BinaryRefAddr)) {
            BinaryRefAddr binaryRefAddr = (BinaryRefAddr) addr;
            byte[] bytes = (byte[]) binaryRefAddr.getContent();

            try {
                return type.cast(Serialization.loadObject(bytes, type.getClassLoader()));
            } catch (IOException e) {
                logger.warn("Unable to load object ''{0}''", key, e);
            } catch (ClassNotFoundException e) {
                logger.warn("Unable to load object ''{0}'' using context classloader", key, e);
            }
        }
        return null;
    }
}
