/**
 * EasyBeans
 * Copyright (C) 2008 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: WebServiceRefBuilder.java 4674 2009-02-20 15:45:11Z sauthieg $
 * --------------------------------------------------------------------------
 */

package org.ow2.util.ee.builder.webserviceref;

import javax.naming.NamingException;
import javax.naming.Reference;

import org.ow2.util.ee.builder.webserviceref.factory.WebServiceRefObjectFactory;
import org.ow2.util.ee.metadata.common.api.struct.IJaxwsWebServiceRef;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Builds a {@link Reference} that may be bound in JNDI, and used to
 * reconstruct the Service object.<br/>
 * This class is used to factorize all common stuff that a valid
 * JAX-WS Service Reference may need, anyway, subclasses are free to add
 * any values of interest in the Reference.<br/>
 * Notice that the subclasses SHOULD NOT insert specific values using the
 * keys defined in this class! BTW, if they do do, these values will be
 * overridden afterwards.
 * @author Guillaume Sauthier
 */
public class WebServiceRefBuilder {

    /**
     * Logger.
     */
    private static Log logger = LogFactory.getLog(WebServiceRefBuilder.class);

    /**
     * Build a JNDI Reference instance that will be capable of returning
     * a valid WebService client view (Service/SEI class).
     * The created Reference must contains (in a serialized or String form)
     * anything that is required to re-create the Service instance.
     *
     * @param serviceRef Metadata used to create the Reference
     * @return a Reference that will be used to re-create a JAX-WS Service.
     * @throws NamingException if reference is not thrown
     */
    public Reference build(final IJaxwsWebServiceRef serviceRef) throws NamingException {

        if (isNullOrEmpty(serviceRef.getValue())) {
            // if the value was not set, we use the type instead
            serviceRef.setValue(serviceRef.getType());
        }

        String referenceClassname = null;
        String serviceClassname = serviceRef.getValue();
        String seiClassname = null;
        if (!serviceRef.getValue().equals(serviceRef.getType())) {
            // Required injection: SEI
            referenceClassname = serviceRef.getType();
            seiClassname = serviceRef.getType();

            logger.debug("Port injection required (Service:''{0}'', SEI:''{1}'')",
                         serviceClassname,
                         seiClassname);
        } else {
            // Required injection: Service
            referenceClassname = serviceRef.getValue();

            logger.debug("Service injection required (Service:''{0}'')",
                         serviceClassname);
        }

        // Let a chance for sub classes to specify its own ObjectFactory
        Reference ref = createReference(referenceClassname);

        ReferenceHelper helper = new ReferenceHelper(ref);

        helper.insert(Constants.SERVICE_CLASS.name(), serviceClassname);
        helper.insert(Constants.SERVICE_ENDPOINT_INTERFACE.name(), seiClassname);
        helper.insert(Constants.WSDL_LOCATION.name(), serviceRef.getWsdlLocation());
        helper.insert(Constants.SERVICE_REF_ANNOTATION.name(), serviceRef);

        // Manage HandlerChain
        if (serviceRef.getHandlerChains() != null) {
            // The user has set a handler chain in its deployment descriptor
            // It has priority over annotations
            helper.insert(Constants.HANDLER_CHAIN_STRUCT.name(), serviceRef.getHandlerChains());
            logger.debug("Build ref using handler-chains from descriptor");
        } else {

            // Try to see if there was an existing annotation along with the @WebServiceRef
            if (serviceRef.getHandlerChainFile() != null) {
                helper.insert(Constants.HANDLER_CHAIN_FILE.name(), serviceRef.getHandlerChainFile());
                helper.insert(Constants.DECLARING_CLASS.name(), serviceRef.getDeclaringClass());

                logger.debug("Build ref using handler-chains from annotation");
            }
        }


        return ref;
    }

    /**
     * Any JAX-WS provider should override this method to set its own ObjectFactory
     * (subclass of WebServiceRefObjectFactory).
     * @param classname classname of the instance that will be returned by the ObjectFactory.
     * @return an empty JNDI Reference bound to a specific ObjectFactory.
     */
    protected Reference createReference(final String classname) {
        return new Reference(classname, WebServiceRefObjectFactory.class.getName(), null);
    }

    /**
     * @param value tested String
     * @return <code>true</code> if the parameter is null or empty.
     */
    private static boolean isNullOrEmpty(final String value) {
        return (value == null) || "".equals(value);
    }

}
