/**
 * EasyBeans
 * Copyright (C) 2008 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: WebServiceRefObjectFactory.java 4804 2009-03-13 09:52:24Z benoitf $
 * --------------------------------------------------------------------------
 */

package org.ow2.util.ee.builder.webserviceref.factory;

import java.lang.reflect.Constructor;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Hashtable;

import javax.naming.Context;
import javax.naming.Name;
import javax.naming.Reference;
import javax.naming.spi.ObjectFactory;
import javax.xml.namespace.QName;
import javax.xml.ws.Service;
import javax.xml.ws.WebServiceClient;

import org.ow2.util.ee.builder.webserviceref.Constants;
import org.ow2.util.ee.builder.webserviceref.ReferenceHelper;
import org.ow2.util.ee.metadata.common.api.struct.IJaxwsWebServiceRef;
import org.ow2.util.ee.metadata.common.impl.struct.JaxwsWebServiceRef;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * This ObjectFactory is dedicated to create a Service/SEI instance from
 * a given JNDI Reference.
 * @author Guillaume Sauthier
 */
public class WebServiceRefObjectFactory implements ObjectFactory {

    /**
     * Logger.
     */
    private static Log logger = LogFactory.getLog(WebServiceRefObjectFactory.class);

    /**
     * Creates an object using the location or reference information
     * specified.
     *
     * @param obj         The possibly null object containing location or reference
     *                    information that can be used in creating an object.
     * @param name        The name of this object relative to <code>nameCtx</code>,
     *                    or null if no name is specified.
     * @param nameCtx     The context relative to which the <code>name</code>
     *                    parameter is specified, or null if <code>name</code> is
     *                    relative to the default initial context.
     * @param environment The possibly null environment that is used in
     *                    creating the object.
     * @return The object created; null if an object cannot be created.
     * @throws Exception if this object factory encountered an exception
     *                   while attempting to create an object, and no other object factories are
     *                   to be tried.
     * @see javax.naming.spi.NamingManager#getObjectInstance
     * @see javax.naming.spi.NamingManager#getURLContext
     */
        public Object getObjectInstance(final Object obj,
                                        final Name name,
                                        final Context nameCtx,
                                        final Hashtable<?, ?> environment) throws Exception {

        if (!(obj instanceof Reference)) {
            // Do not support something that is not a Reference
            return null;
        }
        Reference ref = (Reference) obj;

        ReferenceHelper helper = new ReferenceHelper(ref);

        // Assume that the TCCL is a ClassLoader that can load everything we need
        ClassLoader loader = Thread.currentThread().getContextClassLoader();

        // Extract what we need from the Reference
        String serviceClassname = helper.extract(Constants.SERVICE_CLASS.name());
        String seiClassname = helper.extract(Constants.SERVICE_ENDPOINT_INTERFACE.name());
        String wsdlLocation = helper.extract(Constants.WSDL_LOCATION.name());
        IJaxwsWebServiceRef jaxWebServiceRef = helper.extract(Constants.SERVICE_REF_ANNOTATION.name(), JaxwsWebServiceRef.class);

        // Load the class
        Class<? extends Service> serviceClass = null;
        serviceClass = Class.forName(serviceClassname, true, loader).asSubclass(Service.class);

        // Transform the WSDL location into an URL (if any)
        boolean haveWSDL = true;
        URL location = null;
        if (wsdlLocation != null) {
            try {
                // Try if this is an absolute URL
                location = new URL(wsdlLocation);
            } catch (MalformedURLException mue) {
                // Was not absolute
                // Use the service class's classloader to look up the WSDL URL
                ClassLoader serviceLoader = serviceClass.getClassLoader();
                location = serviceLoader.getResource(wsdlLocation);

                if (location == null) {
                    // OK last chance, try with the TCCL
                    location = loader.getResource(wsdlLocation);
                }

                if (location == null) {
                    // If it's still null at this point, remember it ...
                    haveWSDL = false;
                    // ... and let the JAX-WS Provider load it (JAX-WS 7.2.5)
                }
            }
        } else {
            haveWSDL = false;
        }

        // Look at the @WebServiceClient annotation on the Service generated class (if any)
        // and load the name and targetNamespace attributes that denotes the service's QName

        QName serviceQName = null;
        WebServiceClient annotation = serviceClass.getAnnotation(WebServiceClient.class);
        if (annotation != null) {
            // If we have something, use it
            serviceQName = new QName(annotation.targetNamespace(), annotation.name());

            // WSDL location specified ? use it !
            if (!haveWSDL) {
                String annotationWsdlLocation = annotation.wsdlLocation();
                if (!"".equals(annotationWsdlLocation)) {
                    URL urlWsdlLocation = loader.getResource(annotationWsdlLocation);
                    if (urlWsdlLocation != null) {
                        haveWSDL = true;
                        location = urlWsdlLocation;
                    } else {
                        // not found but specified !
                        logger
                        .warn("Unable to found an URL referencing the WSDL Location ''{0}'' "
                                + "specified in the WebServiceClient annotation ''{1}''", annotationWsdlLocation,
                                annotation.name());
                    }
                }
            }

        }
        // TODO the service QName could be also read from the XML standard descriptor

        // Instance of port processor for auth/mtom
        IPortProcessor portProcessor = new PortProcessorImpl(jaxWebServiceRef);

        // Detect JAX-WS version
        JAXWSVersion version = JAXWSVersion.JAXWS_20;
        try {
            serviceClass.getClassLoader().loadClass("javax.xml.ws.soap.MTOMFeature");
            version = JAXWSVersion.JAXWS_21;
        } catch (Exception e) {
            version = JAXWSVersion.JAXWS_20;
        } catch (Error e) {
            version = JAXWSVersion.JAXWS_20;
        }

        // Generate class (if not defined)
        Class<? extends Service> subServiceClass = SubClassServiceGenerator.getClass(serviceClass, haveWSDL, version);


        // Get the constructor and create the service instance
        Service service = null;
        if (haveWSDL) {
            // Use the constructor that can override the WSDL URL value
            Constructor<? extends Service> constructor = subServiceClass.getDeclaredConstructor(IPortProcessor.class, URL.class,
                    QName.class);

            service = constructor.newInstance(portProcessor, location, serviceQName);
        } else {
            // Use the constructor that use default values
            Constructor<? extends Service> constructor = subServiceClass.getDeclaredConstructor(IPortProcessor.class);
            service = constructor.newInstance(portProcessor);
        }

        // Allow specific ws providers to customize the
        // Service instance before returning it to the client
        processObjectInstance(service, helper);

        // By default, return the service instance
        Object returnedObject = service;

        // Maybe the client requires the SEI, not the Service
        if (seiClassname != null) {
            ClassLoader serviceClassLoader = service.getClass().getClassLoader();
            Class<?> sei = serviceClassLoader.loadClass(seiClassname);
            returnedObject = service.getPort(sei);
        }

        // return to the client
        return returnedObject;
    }

    /**
     * Prepare the service before returning it to the client.
     * This is basically a hook for WS provider specific tasks to do.
     * Like setup HandlerChains, ...
     * Warning, Exceptions handling is like what is done for JNDI
     * ObjectFactory: if an Exception is thrown, no other ObjectFactories
     * are tried and the object will never been build, even if other registered
     * factories may have been able to build an Object from the Reference.
     * So throw Exception with care.
     * @param service Service to be customized
     * @param referenceHelper the Reference used for object creation
     * @throws Exception if preparation cannot be performed.
     */
    protected void processObjectInstance(final Service service,
                                         final ReferenceHelper referenceHelper) throws Exception {
        // Default do nothing special
    }

}
