/**
 * OW2 Util
 * Copyright (C) 2008 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 * --------------------------------------------------------------------------
 * $Id: EnhancedCluePool.java 4809 2009-03-15 10:13:49Z gaellalire $
 * --------------------------------------------------------------------------
 */

package org.ow2.util.pool.impl.enhanced;

import java.util.concurrent.Executor;

import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;
import org.ow2.util.pool.api.IPoolConfiguration;
import org.ow2.util.pool.api.Pool;
import org.ow2.util.pool.api.PoolException;
import org.ow2.util.pool.impl.enhanced.api.IPool;
import org.ow2.util.pool.impl.enhanced.api.TimeoutPoolException;
import org.ow2.util.pool.impl.enhanced.api.clue.IClueManager;
import org.ow2.util.pool.impl.enhanced.api.clue.ICluePool;
import org.ow2.util.pool.impl.enhanced.api.keepbusy.ShareMethod;
import org.ow2.util.pool.impl.enhanced.api.resizer.clue.IResizableCluePool;
import org.ow2.util.pool.impl.enhanced.deprecated.PoolFactory;
import org.ow2.util.pool.impl.enhanced.deprecated.PoolFactoryImpl;
import org.ow2.util.pool.impl.enhanced.deprecated.ProxyLastUsedClueAccessManager;
import org.ow2.util.pool.impl.enhanced.impl.basic.accessmanager.clue.LastUsedClueAccessManager;
import org.ow2.util.pool.impl.enhanced.impl.basic.clue.BasicCluePool;
import org.ow2.util.pool.impl.enhanced.impl.keepbusy.KeepBusyPool;
import org.ow2.util.pool.impl.enhanced.impl.keepbusy.clue.KeepBusyCluePool;
import org.ow2.util.pool.impl.enhanced.impl.limited.LimitedWaiterPool;
import org.ow2.util.pool.impl.enhanced.impl.limited.TooManyWaiterException;
import org.ow2.util.pool.impl.enhanced.impl.limited.clue.LimitedWaiterCluePool;
import org.ow2.util.pool.impl.enhanced.impl.listener.resizer.clue.StatsResizerCluePool;
import org.ow2.util.pool.impl.enhanced.impl.listener.statistics.BasicPoolStats;
import org.ow2.util.pool.impl.enhanced.impl.listener.statistics.clue.BasicCluePoolStatsFillerPoolListener;
import org.ow2.util.pool.impl.enhanced.impl.util.ExecutorProvider;
import org.ow2.util.pool.impl.enhanced.impl.validator.clue.ValidatorCluePool;
import org.ow2.util.pool.impl.enhanced.internal.resizer.api.IResizer;
import org.ow2.util.pool.impl.enhanced.internal.resizer.api.shared.ISharedManager;
import org.ow2.util.pool.impl.enhanced.internal.resizer.impl.AsynchronousResizer;
import org.ow2.util.pool.impl.enhanced.internal.resizer.impl.shared.SharedResizer;
import org.ow2.util.pool.impl.enhanced.internal.resizer.impl.stocker.MinMaxExpectedSpareStockerResizer;
import org.ow2.util.pool.impl.enhanced.manager.clue.ICluePoolManager;
import org.ow2.util.pool.impl.enhanced.manager.clue.optional.IValidatorCluePoolManager;

/**
 *
 * @author Gael Lalire
 * @param <E>
 * @param <C>
 */
public class EnhancedCluePool<E, C> implements Pool<E, C> {

    private static final Log LOG = LogFactory.getLog(EnhancedCluePool.class);

    private ICluePool<E, C> lastCluePool;

    private IPool<E> lastPool;

    private LimitedWaiterPool<E> limitedWaiterPool;

    private KeepBusyPool<E> keepBusyPool;

    private MinMaxExpectedSpareStockerResizer<BasicPoolStats> minMaxExpectedSparePoolResizer;

    private long timeout;

    private int maxPool = 0;

    public EnhancedCluePool(final ICluePoolManager<E, C> cluePoolManager) {
        this(cluePoolManager, ResizerType.SIMPLE, null, ExecutorProvider.NEW_THREAD_EXECUTOR);
    }

    public EnhancedCluePool(final ICluePoolManager<E, C> cluePoolManager, final ResizerType resizerType, final ISharedManager sharedManager, final Executor createItemExecutor) {
        LimitedWaiterCluePool<E, C> limitedWaiterCluePool;
        KeepBusyCluePool<E, C> keepBusyCluePool;
        StatsResizerCluePool<E, C, BasicPoolStats> statsResizerCluePool;
        ValidatorCluePool<E, C> validatorCluePool;
        IResizableCluePool<E, C> cluePool;

        ICluePool<E, C> currentCluePool;
        // 1 clue pool
        cluePool = new BasicCluePool<E, C>(cluePoolManager, 0, new LastUsedClueAccessManager<E, C>(cluePoolManager),
                createItemExecutor);

        minMaxExpectedSparePoolResizer = new MinMaxExpectedSpareStockerResizer<BasicPoolStats>(0, 0, 150, true);

        IResizer<BasicPoolStats> resizer = minMaxExpectedSparePoolResizer;

        switch (resizerType) {
        case ASYNCHRONOUS:
            resizer = new AsynchronousResizer<BasicPoolStats>(minMaxExpectedSparePoolResizer);
            break;
        default:
        case SHARED_ASYNCHRONOUS:
            resizer = new SharedResizer<BasicPoolStats>(sharedManager, minMaxExpectedSparePoolResizer);
            break;
        case SIMPLE:
            break;
        }
        // IResizer<BasicPoolStats> resizer = minMaxExpectedSparePoolResizer;
        resizer.setResizable(cluePool);
        currentCluePool = cluePool;

        // 2 validator
        if (IValidatorCluePoolManager.class.isInstance(cluePoolManager)) {
            IValidatorCluePoolManager<E, C> validatorCluePoolManager = (IValidatorCluePoolManager<E, C>) cluePoolManager;
            validatorCluePool = new ValidatorCluePool<E, C>(currentCluePool, validatorCluePoolManager, false);
            currentCluePool = validatorCluePool;
        }

        // 3.0 stats & resize pool
        statsResizerCluePool = new StatsResizerCluePool<E, C, BasicPoolStats>(currentCluePool,
                new BasicCluePoolStatsFillerPoolListener<E, C>(), new BasicPoolStats(), resizer);
        // 3.1 add a 3e listener
        // FIXME manage remove listener and activate / passivate listener => create new optional managers
        // statsResizerCluePool.addCluePoolListener(poolFactoryImpl);
        currentCluePool = statsResizerCluePool;

        // 4 keep busy pool
        keepBusyCluePool = new KeepBusyCluePool<E, C>(currentCluePool, cluePoolManager);
        // 5 limited waiter pool
        limitedWaiterCluePool = new LimitedWaiterCluePool<E, C>(keepBusyCluePool);
        currentCluePool = limitedWaiterCluePool;
        // end pool construction

        limitedWaiterPool = limitedWaiterCluePool;
        keepBusyPool = keepBusyCluePool;
        lastCluePool = currentCluePool;
        lastPool = currentCluePool;

        timeout = 0;
        setAllowSharedInstance(false);
        setPoolConfiguration(new PoolConfiguration());
    }

    /**
     * This method exists to keep a compatibility with previous
     * ow2 pool.
     */
    @Deprecated
    public EnhancedCluePool(final PoolFactory<E, C> poolFactory, final IClueManager<E, C> clueManager) {
        this(poolFactory, ResizerType.SIMPLE, null, clueManager, ExecutorProvider.NEW_THREAD_EXECUTOR);
    }

    @Deprecated
    public EnhancedCluePool(final PoolFactory<E, C> poolFactory, final ResizerType resizerType, final ISharedManager sharedManager, final IClueManager<E, C> clueManager, final Executor createItemExecutor) {
        LimitedWaiterCluePool<E, C> limitedWaiterCluePool;
        KeepBusyCluePool<E, C> keepBusyCluePool;
        StatsResizerCluePool<E, C, BasicPoolStats> statsResizerCluePool;
        ValidatorCluePool<E, C> validatorCluePool;
        IResizableCluePool<E, C> cluePool;

        PoolFactoryImpl<E, C> poolFactoryImpl = new PoolFactoryImpl<E, C>(poolFactory);

        // 1 clue pool
        cluePool = new BasicCluePool<E, C>(poolFactoryImpl, 0, new ProxyLastUsedClueAccessManager<E, C>(poolFactoryImpl,
                clueManager), createItemExecutor);

        minMaxExpectedSparePoolResizer = new MinMaxExpectedSpareStockerResizer<BasicPoolStats>(0, 0, 150, true);

        IResizer<BasicPoolStats> resizer = minMaxExpectedSparePoolResizer;

        switch (resizerType) {
        case ASYNCHRONOUS:
            resizer = new AsynchronousResizer<BasicPoolStats>(minMaxExpectedSparePoolResizer);
            break;
        default:
        case SHARED_ASYNCHRONOUS:
            resizer = new SharedResizer<BasicPoolStats>(sharedManager, minMaxExpectedSparePoolResizer);
            break;
        case SIMPLE:
            break;
        }
        // IResizer<BasicPoolStats> resizer = minMaxExpectedSparePoolResizer;
        resizer.setResizable(cluePool);

        // 2 validator
        validatorCluePool = new ValidatorCluePool<E, C>(cluePool, poolFactoryImpl, false);

        // 3.0 stats & resize pool
        statsResizerCluePool = new StatsResizerCluePool<E, C, BasicPoolStats>(validatorCluePool,
                new BasicCluePoolStatsFillerPoolListener<E, C>(), new BasicPoolStats(), resizer);
        // 3.1 add a 3e listener
        statsResizerCluePool.addCluePoolListener(poolFactoryImpl);

        // 4 keep busy pool
        keepBusyCluePool = new KeepBusyCluePool<E, C>(statsResizerCluePool, clueManager);
        // 5 limited waiter pool
        limitedWaiterCluePool = new LimitedWaiterCluePool<E, C>(keepBusyCluePool);
        // end pool construction

        limitedWaiterPool = limitedWaiterCluePool;
        keepBusyPool = keepBusyCluePool;
        lastCluePool = limitedWaiterCluePool;
        lastPool = limitedWaiterCluePool;

        timeout = 0;
        setAllowSharedInstance(false);
        setPoolConfiguration(new PoolConfiguration());
    }


    public E get() throws PoolException {
        try {
            LOG.debug("START GET {1} try to get with timeout {0}", timeout);
            E e = lastPool.get(timeout);
            LOG.debug("OK FOR GET {1} try to get with timeout {0}", timeout);
            return e;
        } catch (TimeoutPoolException e) {
            throw new PoolException("No more instances available", e);
        } catch (TooManyWaiterException e) {
            throw new PoolException("No more instances available", e);
        } catch (org.ow2.util.pool.impl.enhanced.api.PoolException e) {
            throw new PoolException("Unknow exception", e);
        }
    }

    public E get(final C clue) throws PoolException {
        try {
            LOG.debug("try to get with clue {0} and timeout {1}", clue, timeout);
            return lastCluePool.get(clue, timeout);
        } catch (TimeoutPoolException e) {
            throw new PoolException("No more instances available", e);
        } catch (TooManyWaiterException e) {
            throw new PoolException("No more instances available", e);
        } catch (org.ow2.util.pool.impl.enhanced.api.PoolException e) {
            throw new PoolException("Unknow exception", e);
        }

    }

    public void discard(final E instance) throws PoolException {
        LOG.debug("discard {0} ", instance);
        try {
            lastPool.remove(instance);
        } catch (org.ow2.util.pool.impl.enhanced.api.PoolException e) {
            throw new PoolException("Unknow exception", e);
        }
    }

    public void release(final E instance) throws PoolException {
        LOG.debug("release {0} ", instance);
        try {
            lastPool.put(instance);
        } catch (org.ow2.util.pool.impl.enhanced.api.PoolException e) {
            throw new PoolException("Unknow exception", e);
        }
    }

    public void setPoolConfiguration(final IPoolConfiguration poolConfiguration) {
        int maxPool = poolConfiguration.getMax();
        timeout = poolConfiguration.getTimeout();
        limitedWaiterPool.setMaxWaiter(poolConfiguration.getMaxWaiters());
        this.maxPool = maxPool;
        minMaxExpectedSparePoolResizer.setMaxPool(maxPool);
        minMaxExpectedSparePoolResizer.update();
        LOG.debug("{3} new conf Setted (timeout : {0}, maxwaiter: {1}, max: {2})", timeout, poolConfiguration.getMaxWaiters(),
                maxPool, this);
    }

    /**
     * Allow to share instances (shouldn't be use in stateful case as one
     * stateful ID is linked to one client. No Stateful with same ID are
     * authorized).
     * @param allowSharedInstance true if it is allowed, else false.
     */
    public void setAllowSharedInstance(final boolean allowSharedInstance) {
        if (allowSharedInstance) {
            keepBusyPool.setShareMethod(ShareMethod.SHARE_AS_LAST_SOLUTION);
        } else {
            keepBusyPool.setShareMethod(ShareMethod.NEVER_SHARE);
        }
    }

    /**
     * Return true if many instances of the same object can be created (when
     * pool is not full).
     * @return true if this is allowed.
     */
    public boolean isAllowSharedInstance() {
        return keepBusyPool.getShareMethod() != ShareMethod.NEVER_SHARE;
    }

    public void start() throws PoolException {
        minMaxExpectedSparePoolResizer.setMaxPool(maxPool);
    }

    public void stop() throws PoolException {
        minMaxExpectedSparePoolResizer.setMaxPool(0);
        lastPool.interruptAllWaiters();
    }

}
