/**
 * OW2 Util
 * Copyright (C) 2009 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 * --------------------------------------------------------------------------
 * $Id: EnhancedCluePoolFactory.java 4809 2009-03-15 10:13:49Z gaellalire $
 * --------------------------------------------------------------------------
 */

package org.ow2.util.pool.impl.enhanced;

import java.util.concurrent.Executor;

import javax.resource.spi.work.WorkManager;

import org.ow2.util.pool.impl.enhanced.api.IPool;
import org.ow2.util.pool.impl.enhanced.api.thread.IReusableThread;
import org.ow2.util.pool.impl.enhanced.impl.basic.BasicPool;
import org.ow2.util.pool.impl.enhanced.impl.basic.accessmanager.LastUsedAccessManager;
import org.ow2.util.pool.impl.enhanced.impl.thread.ThreadPool;
import org.ow2.util.pool.impl.enhanced.impl.thread.managementthread.ManagementThreadReusableThreadFactory;
import org.ow2.util.pool.impl.enhanced.impl.thread.workmanager.WorkManagerReusableThreadFactory;
import org.ow2.util.pool.impl.enhanced.impl.util.ExecutorProvider;
import org.ow2.util.pool.impl.enhanced.internal.resizer.impl.shared.SharedResizerPoolThreadManager;
import org.ow2.util.pool.impl.enhanced.manager.clue.ICluePoolManager;

/**
 *
 * @author Gael Lalire
 */
public class EnhancedCluePoolFactory {

    /**
     * Pool of reusable thread.
     */
    private IPool<IReusableThread> pool;

    /**
     * Use a management thread to create new pool item (Executor).
     * Use a management thread also to call setExpectedSize (ISharedManager).
     * @param poolSize the number of management thread
     */
    public EnhancedCluePoolFactory(final int poolSize) {
        pool = new BasicPool<IReusableThread>(new ManagementThreadReusableThreadFactory(true), poolSize,
                new LastUsedAccessManager<IReusableThread>(), ExecutorProvider.SELF_THREAD_EXECUTOR);
    }

    /**
     * Use the work manager to create, fetch and record items of pool (Executor).
     * Use the work manager also to call setExpectedSize (ISharedManager).
     * @param workManager the workManager
     * @param limit max call to doSchedule of pool management
     */
    public EnhancedCluePoolFactory(final WorkManager workManager, final int limit) {
        pool = new BasicPool<IReusableThread>(new WorkManagerReusableThreadFactory(workManager),
                limit, new LastUsedAccessManager<IReusableThread>(), ExecutorProvider.SELF_THREAD_EXECUTOR);
    }

    /**
     * Use the executor to create, fetch and record items of pool (Executor).
     * Use the executor also to call setExpectedSize (ISharedManager).
     * @param executor the executor
     * @param limit max call to execute of pool management
     */
    public EnhancedCluePoolFactory(final Executor executor, final int limit) {
        pool = new BasicPool<IReusableThread>(new ManagementThreadReusableThreadFactory(
                true), limit, new LastUsedAccessManager<IReusableThread>(), ExecutorProvider.SELF_THREAD_EXECUTOR);

    }

    /**
     * You should use this method to create {@link EnhancedCluePool}.
     * @param <E> pool item type
     * @param <C> clue type
     * @param cluePoolManager a clue pool manager
     * @return a new clue pool
     */
    public <E, C> EnhancedCluePool<E, C> createEnhancedCluePool(final ICluePoolManager<E, C> cluePoolManager) {
        return createEnhancedCluePool(cluePoolManager, false, true);
    }

    /**
     *
     * @param <E> pool item type
     * @param <C> clue type
     * @param cluePoolManager a clue pool manager
     * @param setSizeShared if true setExpectiveSize will be called in a separate thread, you
     * should prefers the schedulerShared option because on a remove call the creation
     * will be on self thread.
     * @param schedulerShared each time a pool item is needed a shared executor will be
     * used
     * @return a new clue pool
     */
    public <E, C> EnhancedCluePool<E, C> createEnhancedCluePool(final ICluePoolManager<E, C> cluePoolManager,
            final boolean setSizeShared, final boolean schedulerShared) {
        Executor executor;
        if (schedulerShared) {
            executor = new ThreadPool(pool);
        } else {
            executor = ExecutorProvider.SELF_THREAD_EXECUTOR;
        }
        if (setSizeShared) {
            return new EnhancedCluePool<E, C>(cluePoolManager, ResizerType.SHARED_ASYNCHRONOUS,
                    new SharedResizerPoolThreadManager(pool), executor);
        } else {
            return new EnhancedCluePool<E, C>(cluePoolManager, ResizerType.SIMPLE, null, executor);
        }

    }


}
