/**
 * OW2 Util
 * Copyright (C) 2008 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 * --------------------------------------------------------------------------
 * $Id: EnhancedPool.java 4809 2009-03-15 10:13:49Z gaellalire $
 * --------------------------------------------------------------------------
 */

package org.ow2.util.pool.impl.enhanced;

import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;
import org.ow2.util.pool.api.IPoolConfiguration;
import org.ow2.util.pool.api.PoolException;
import org.ow2.util.pool.impl.enhanced.api.IPool;
import org.ow2.util.pool.impl.enhanced.api.TimeoutPoolException;
import org.ow2.util.pool.impl.enhanced.api.keepbusy.ShareMethod;
import org.ow2.util.pool.impl.enhanced.impl.keepbusy.KeepBusyPool;
import org.ow2.util.pool.impl.enhanced.impl.limited.LimitedWaiterPool;
import org.ow2.util.pool.impl.enhanced.impl.limited.TooManyWaiterException;
import org.ow2.util.pool.impl.enhanced.impl.listener.statistics.BasicPoolStats;
import org.ow2.util.pool.impl.enhanced.internal.resizer.impl.stocker.MinMaxExpectedSpareStockerResizer;
import org.ow2.util.pool.impl.enhanced.manager.IPoolManager;

/**
 *
 * @author Gael Lalire
 * @param <E>
 */
public class EnhancedPool<E> {

    private static final Log LOG = LogFactory.getLog(EnhancedPool.class);

    private IPool<E> lastPool;

    private LimitedWaiterPool<E> limitedWaiterPool;

    private KeepBusyPool<E> keepBusyPool;

    private MinMaxExpectedSpareStockerResizer<BasicPoolStats> minMaxExpectedSparePoolResizer;

    private long timeout;

    private int maxPool = 0;

    public EnhancedPool(final IPoolManager<E> poolManager) {
        this(poolManager, ResizerType.SHARED_ASYNCHRONOUS);
    }

    public EnhancedPool(final IPoolManager<E> poolManager, final ResizerType resizerType) {
        // TODO write it.
        throw new UnsupportedOperationException();
    }

    public E get() throws PoolException {
        try {
            LOG.debug("START GET {1} try to get with timeout {0}", timeout);
            E e = lastPool.get(timeout);
            LOG.debug("OK FOR GET {1} try to get with timeout {0}", timeout);
            return e;
        } catch (TimeoutPoolException e) {
            throw new PoolException("No more instances available", e);
        } catch (TooManyWaiterException e) {
            throw new PoolException("No more instances available", e);
        } catch (org.ow2.util.pool.impl.enhanced.api.PoolException e) {
            throw new PoolException("Unknow exception", e);
        }
    }

    public void discard(final E instance) throws PoolException {
        LOG.debug("discard {0} ", instance);
        try {
            lastPool.remove(instance);
        } catch (org.ow2.util.pool.impl.enhanced.api.PoolException e) {
            throw new PoolException("Unknow exception", e);
        }
    }

    public void release(final E instance) throws PoolException {
        LOG.debug("release {0} ", instance);
        try {
            lastPool.put(instance);
        } catch (org.ow2.util.pool.impl.enhanced.api.PoolException e) {
            throw new PoolException("Unknow exception", e);
        }
    }

    public void setPoolConfiguration(final IPoolConfiguration poolConfiguration) {
        int maxPool = poolConfiguration.getMax();
        timeout = poolConfiguration.getTimeout();
        limitedWaiterPool.setMaxWaiter(poolConfiguration.getMaxWaiters());
        this.maxPool = maxPool;
        minMaxExpectedSparePoolResizer.setMaxPool(maxPool);
        minMaxExpectedSparePoolResizer.update();
        LOG.debug("{3} new conf Setted (timeout : {0}, maxwaiter: {1}, max: {2})", timeout, poolConfiguration.getMaxWaiters(),
                maxPool, this);
    }

    /**
     * Allow to share instances (shouldn't be use in stateful case as one
     * stateful ID is linked to one client. No Stateful with same ID are
     * authorized).
     * @param allowSharedInstance true if it is allowed, else false.
     */
    public void setAllowSharedInstance(final boolean allowSharedInstance) {
        if (allowSharedInstance) {
            keepBusyPool.setShareMethod(ShareMethod.SHARE_AS_LAST_SOLUTION);
        } else {
            keepBusyPool.setShareMethod(ShareMethod.NEVER_SHARE);
        }
    }

    /**
     * Return true if many instances of the same object can be created (when
     * pool is not full).
     * @return true if this is allowed.
     */
    public boolean isAllowSharedInstance() {
        return keepBusyPool.getShareMethod() != ShareMethod.NEVER_SHARE;
    }

    public void start() throws PoolException {
        minMaxExpectedSparePoolResizer.setMaxPool(maxPool);
    }

    public void stop() throws PoolException {
        minMaxExpectedSparePoolResizer.setMaxPool(0);
        lastPool.interruptAllWaiters();
    }

}
