/**
 * OW2 Util
 * Copyright (C) 2008 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 * --------------------------------------------------------------------------
 * $Id: PoolConfiguration.java 4389 2008-12-15 13:48:57Z alitokmen $
 * --------------------------------------------------------------------------
 */

package org.ow2.util.pool.impl.enhanced;

import java.io.Serializable;

import org.ow2.util.pool.api.IPoolConfiguration;

/**
 * Implementation of the Pool Configuration interface.
 * @author Florent BENOIT
 */
public class PoolConfiguration implements IPoolConfiguration, Serializable {

    /**
     * Id for serializable class.
     */
    private static final long serialVersionUID = 3409409761151246408L;

    /**
     * Maximum size of the pool.
     */
    private int max = DEFAULT_MAX;

    /**
     * Timeout for waiters that are waiting for an empty element in the pool.
     */
    private long timeout = DEFAULT_TIMEOUT;

    /**
     * Number of waiters that can wait for the given timeout.
     */
    private int maxWaiters = DEFAULT_MAX_WAITERS;

    /**
     * @return Maximum size of the pool.
     */
    public int getMax() {
        return max;
    }

    /**
     * Sets the Maximum size of the pool.
     * @param max the given maximum size of the pool
     */
    public void setMax(final int max) {
        this.max = max;
    }

    /**
     * @return the timeout for waiters that are waiting for an empty element in the pool.
     */
    public long getTimeout() {
        return timeout;
    }

    /**
     * Sets the timeout for waiters that are waiting for an empty element in the pool..
     * @param timeout the timeout for waiters that are waiting for an empty element in the pool.
     */
    public void setTimeout(final long timeout) {
        this.timeout = timeout;
    }

    /**
     * @return the number of waiters that can wait for the given timeout. If this number is
     * reached, there is no wait and an exception is thrown.
     */
    public int getMaxWaiters() {
        return maxWaiters;
    }

    /**
     * Sets the number of waiters that can wait for the given timeout.
     * @param maxWaiters the number of waiters that can wait for the given timeout.
     */
    public void setMaxWaiters(final int maxWaiters) {
        this.maxWaiters = maxWaiters;
    }

    /**
     * Indicates whether some other object is "equal to" this one.
     * @param  object the reference object with which to compare.
     * @return  <code>true</code> if this object is the same as the obj
     *          argument; <code>false</code> otherwise.
     */
    @Override
    public boolean equals(final Object object) {
        if (object == null || !(object instanceof PoolConfiguration)) {
            return false;
        }
        PoolConfiguration poolConfiguration = (PoolConfiguration) object;
        return max == poolConfiguration.max
            && maxWaiters == poolConfiguration.maxWaiters
            && timeout == poolConfiguration.timeout;
    }

    /**
     * Returns a hash code value for the pool configuration.
     * @return  a hash code value for this pool configuration.
     */
    @Override
    public int hashCode() {
        return (int) (max + maxWaiters + timeout);
    }

}
