/**
 * OW2 Util
 * Copyright (C) 2008 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 * --------------------------------------------------------------------------
 * $Id: BasicPool.java 4809 2009-03-15 10:13:49Z gaellalire $
 * --------------------------------------------------------------------------
 */

package org.ow2.util.pool.impl.enhanced.impl.basic;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.Executor;
import java.util.concurrent.locks.Lock;

import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;
import org.ow2.util.pool.impl.enhanced.api.IPoolItemFactory;
import org.ow2.util.pool.impl.enhanced.api.IllegalTimeoutException;
import org.ow2.util.pool.impl.enhanced.api.NotABusyPoolItemException;
import org.ow2.util.pool.impl.enhanced.api.TimeoutPoolException;
import org.ow2.util.pool.impl.enhanced.api.WaiterInterruptedException;
import org.ow2.util.pool.impl.enhanced.api.basic.IBasicPool;
import org.ow2.util.pool.impl.enhanced.api.basic.NoBusyPoolItemException;
import org.ow2.util.pool.impl.enhanced.api.basic.accessmanager.IAccessManager;
import org.ow2.util.pool.impl.enhanced.impl.util.LockFactory;
import org.ow2.util.pool.impl.enhanced.internal.actionscheduler.ActionSchedulerFactory;
import org.ow2.util.pool.impl.enhanced.internal.actionscheduler.IActionScheduler;
import org.ow2.util.pool.impl.enhanced.internal.lock.api.ISignalClearableCondition;
import org.ow2.util.pool.impl.enhanced.internal.lock.impl.SignalClearableConditionProxy;

/**
 * Basic pool implementation. You can dynamically change pool size calling
 * setExpectedSize method.
 * @author Gael Lalire
 * @param <E> pool item type
 */
public class BasicPool<E> implements IBasicPool<E> {

    private static final Log LOG = LogFactory.getLog(BasicPool.class);

    private Lock lock;

    private ISignalClearableCondition signalClearableCondition;

    private int usedPoolItem;

    private List<E> availablePoolItemList;

    private List<E> unmodifiableAvailablePoolItemList;

    /**
     * @return the unmodifiableAvailablePoolItemList
     */
    protected List<E> getUnmodifiableAvailablePoolItemList() {
        return unmodifiableAvailablePoolItemList;
    }

    private IPoolItemFactory<? extends E> poolItemFactory;

    private int expectedSize;

    private IAccessManager<? super E> accessManager;

    private Object sizeMutex;

    private int waiterCount;

    private boolean interruptingAllWaiters;

    private int scheduledCreateItemCount;

    private IActionScheduler createItemActionScheduler;

    private int delayedCount;

    public BasicPool(final IPoolItemFactory<? extends E> poolItemFactory, final int initPoolSize,
            final IAccessManager<? super E> accessManager, final Executor createItemExecutor) {
        if (initPoolSize < 0 || poolItemFactory == null || accessManager == null) {
            throw new IllegalArgumentException();
        }
        lock = LockFactory.createLock();
        signalClearableCondition = SignalClearableConditionProxy.createProxy(lock.newCondition());
        sizeMutex = new Object();
        scheduledCreateItemCount = 0;
        delayedCount = 0;
        waiterCount = 0;
        interruptingAllWaiters = false;
        availablePoolItemList = new ArrayList<E>();
        unmodifiableAvailablePoolItemList = Collections.unmodifiableList(availablePoolItemList);
        this.accessManager = accessManager;
        this.poolItemFactory = poolItemFactory;
        expectedSize = initPoolSize;
        for (int i = 0; i < initPoolSize; i++) {
            E poolItem = poolItemFactory.createPoolItem();
            if (poolItem == null) {
                throw new NullPointerException("poolItemFactory returns a null instance");
            }
            availablePoolItemList.add(poolItem);
        }
        usedPoolItem = 0;
        createItemActionScheduler = new ActionSchedulerFactory().createActionScheduler(new Callable<Boolean>() {

            public Boolean call() throws Exception {
                return createNewItem();
            }

        }, createItemExecutor);
    }

    /**
     * @param expectedSize the expectedSize to set
     */
    public void setExpectedSize(final int expectedSize) {
        if (expectedSize < 0) {
            throw new IllegalArgumentException();
        }
        LOG.debug("Want mutex for setting size to {0}", expectedSize);
        int instanceToCreate = 0;

        synchronized (sizeMutex) {
            lock.lock();
            try {
                if (this.expectedSize != expectedSize) {
                    LOG.debug("size will change from {0} to {1}", this.expectedSize, expectedSize);
                    this.expectedSize = expectedSize;
                }
                // calcul instance to create (or delete)
                int unUsedPoolItem = availablePoolItemList.size();
                int total = usedPoolItem + unUsedPoolItem;
                instanceToCreate = expectedSize - total;
                if (instanceToCreate < 0) {
                    // we have to many instances (but we cannot delete busy
                    // instances)
                    int instanceToDelete = Math.min(-instanceToCreate, unUsedPoolItem);
                    for (int i = 0; i < instanceToDelete; i++) {
                        availablePoolItemList.remove(accessManager.choosePoolItemToRelease(unmodifiableAvailablePoolItemList));
                    }
                }
            } finally {
                lock.unlock();
            }

            // we have one less instance to create per factory working
            instanceToCreate -= scheduledCreateItemCount;
            if (instanceToCreate <= 0) {
                // try to cancel useless creation
                int cancelled = createItemActionScheduler.cancelAction(-instanceToCreate);
                // remove canceled from scheduled count
                scheduledCreateItemCount -= cancelled;
                return;
            }
            scheduledCreateItemCount += instanceToCreate;
        }
        // ask to call 'instanceToCreate' times the createNewItem() method
        createItemActionScheduler.scheduleAction(instanceToCreate);
    }

    /**
     * @return true if the call to poolItemFactory was done false otherwise
     */
    protected boolean createNewItem() {
        int instanceToCreate = 0;

        synchronized (sizeMutex) {
            lock.lock();
            try {
                // calcul instance to create (or delete)
                int unUsedPoolItem = availablePoolItemList.size();
                int total = usedPoolItem + unUsedPoolItem;
                instanceToCreate = expectedSize - total;
                if (instanceToCreate <= 0) {
                    // we have to many instances (however we should manage
                    // remove)
                    if (instanceToCreate != 0 && unUsedPoolItem != 0) {
                        LOG.error("Issue in algorithm : check remove / put methods");
                    }
                    // creation is useless (cancel operation was called too
                    // late)
                    scheduledCreateItemCount--;
                    return false;
                }
            } finally {
                lock.unlock();
            }
        }

        E poolItem = poolItemFactory.createPoolItem();
        LOG.debug("create new item");

        synchronized (sizeMutex) {

            lock.lock();
            try {
                availablePoolItemList.add(accessManager.createPoolItem(unmodifiableAvailablePoolItemList, poolItem), poolItem);
                // calcul instance to create (or delete)
                int unUsedPoolItem = availablePoolItemList.size();
                int total = usedPoolItem + unUsedPoolItem;
                instanceToCreate = this.expectedSize - total;
                if (instanceToCreate < 0) {
                    // we just create an useless instance
                    int instanceToDelete = Math.min(-instanceToCreate, unUsedPoolItem);
                    if (instanceToDelete != 1) {
                        LOG.error("Issue in algorithm : check remove / put methods");
                    }
                    availablePoolItemList.remove(accessManager.choosePoolItemToRelease(unmodifiableAvailablePoolItemList));
                } else {
                    // our new instance can be use
                    signalClearableCondition.signal();
                }
                if (delayedCount != 0) {
                    // at least a thread wait while some item are available
                    // the reason of that is the DELAY_GET, we have to signalAll
                    // thread in this case
                    // however thanks to lock implementation (clearAllSignal)
                    // all thread will not necessary be wake up.
                    signalClearableCondition.signalAll();
                }

            } finally {
                lock.unlock();
            }
            scheduledCreateItemCount--;
        }
        return true;

    }

    /**
     * This method block is no instance are available.
     * @param timeout after this timeout a exception is thrown
     * @return a pool item
     * @throws TimeoutPoolException if timeout
     */
    public E get(final long timeout) throws TimeoutPoolException, IllegalTimeoutException, WaiterInterruptedException {
        if (timeout < 0 && timeout != INFINITE_TIMEOUT) {
            throw new IllegalTimeoutException();
        }
        E poolItem = null;
        boolean delayed = false;
        lock.lock();
        try {
            int size = availablePoolItemList.size();
            int poolItemToGetChoosed = IAccessManager.DELAY_GET;
            if (size != 0) {
                poolItemToGetChoosed = accessManager.choosePoolItemToGet(unmodifiableAvailablePoolItemList);
                if (poolItemToGetChoosed == IAccessManager.DELAY_GET) {
                    delayedCount++;
                    delayed = true;
                }
            }
            if (poolItemToGetChoosed == IAccessManager.DELAY_GET) {
                long expireTime = 0;
                long timeToWait;
                if (timeout == INFINITE_TIMEOUT) {
                    timeToWait = 0;
                } else {
                    long currentTime = System.currentTimeMillis();
                    expireTime = currentTime + timeout;
                    timeToWait = expireTime - currentTime;
                    if (timeToWait <= 0) {
                        throw new TimeoutPoolException();
                    }
                }
                while (poolItemToGetChoosed == IAccessManager.DELAY_GET) {
                    if (interruptingAllWaiters) {
                        throw new WaiterInterruptedException();
                    }
                    waiterCount++;
                    LOG.debug("nb waiter increase to {0}", waiterCount);
                    try {
                        signalClearableCondition.await(timeToWait);
                        LOG.debug("wait notified or expired");
                    } catch (InterruptedException e) {
                        LOG.debug("wait interrupted");
                    }
                    waiterCount--;
                    LOG.debug("nb waiter decrease to {0}", waiterCount);
                    if (interruptingAllWaiters && waiterCount == 0) {
                        signalClearableCondition.signal();
                    }
                    size = availablePoolItemList.size();
                    if (size != 0) {
                        poolItemToGetChoosed = accessManager.choosePoolItemToGet(unmodifiableAvailablePoolItemList);
                        if (!delayed && poolItemToGetChoosed == IAccessManager.DELAY_GET) {
                            delayedCount++;
                            delayed = true;
                        }
                    } else {
                        if (delayed) {
                            // presume not delayed if no item available
                            delayedCount--;
                            delayed = false;
                        }
                        if (timeout != INFINITE_TIMEOUT) {
                            timeToWait = expireTime - System.currentTimeMillis();
                            if (timeToWait <= 0) {
                                throw new TimeoutPoolException();
                            }
                        }
                    }
                }
            }
            usedPoolItem++;
            poolItem = availablePoolItemList.remove(poolItemToGetChoosed);
            if (size == 1) {
                // no more item to distribute so we avoid wake up of other
                // threads.
                signalClearableCondition.clearAllSignal();
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug("pool {0}/{1}", usedPoolItem, availablePoolItemList.size() + usedPoolItem);
            }
        } finally {
            if (delayed) {
                delayedCount--;
            }
            lock.unlock();
        }
        return poolItem;
    }

    /**
     * @return a non null instance
     * @throws WaiterInterruptedException if error occurs
     */
    public E get() throws WaiterInterruptedException {
        try {
            return get(INFINITE_TIMEOUT);
        } catch (TimeoutPoolException e) {
            throw new Error("get(INFINITE_TIMEOUT) should not return a TimeoutPoolException");
        } catch (IllegalTimeoutException e) {
            throw new Error("get(INFINITE_TIMEOUT) should not return a IllegalTimeoutException");
        }
    }

    /**
     * This method will never block. If the pool owns too many instances,
     * release method will be apply on pool item.
     * @param poolItem a pool item
     * @throws NotABusyPoolItemException if error occurs
     */
    public void put(final E poolItem) throws NotABusyPoolItemException {
        if (poolItem == null) {
            throw new NotABusyPoolItemException();
        }
        lock.lock();
        try {
            if (usedPoolItem <= 0) {
                throw new NoBusyPoolItemException();
            }
            int remainToDelete = usedPoolItem - expectedSize;
            usedPoolItem--;
            availablePoolItemList.add(accessManager.putPoolItem(unmodifiableAvailablePoolItemList, poolItem), poolItem);
            if (remainToDelete <= 0) {
                // add to available if no remain to delete
                if (delayedCount == 0) {
                    signalClearableCondition.signal();
                } else {
                    // at least a thread wait while some item are available
                    // the reason of that is the DELAY_GET, we have to signalAll
                    // thread in this case
                    // however thanks to lock implementation (clearAllSignal)
                    // all thread will not necessary be wake up.
                    signalClearableCondition.signalAll();
                }
            } else {
                // let gc run on poolItem otherwise
                availablePoolItemList.remove(accessManager.choosePoolItemToRelease(unmodifiableAvailablePoolItemList));
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug("pool {0}/{1} (used/total)", usedPoolItem, availablePoolItemList.size() + usedPoolItem);
            }
        } finally {
            lock.unlock();
        }
    }

    /**
     * @return the poolItemFactory
     */
    protected IPoolItemFactory<? extends E> getPoolItemFactory() {
        return poolItemFactory;
    }

    /**
     * @return the releaseListener
     */
    protected IAccessManager<? super E> getReleaseListener() {
        return accessManager;
    }

    /**
     * @param usedPoolItem the usedPoolItem to set
     */
    protected void setUsedPoolItem(final int usedPoolItem) {
        this.usedPoolItem = usedPoolItem;
    }

    /**
     * @return the mutex to sync before calling protected methods
     */
    protected List<E> getAvailablePoolItemList() {
        return availablePoolItemList;
    }

    /**
     * @return the expectedSize
     */
    protected int getExpectedSize() {
        return expectedSize;
    }

    /**
     * @return the usedPoolItem
     */
    protected int getUsedPoolItem() {
        return usedPoolItem;
    }

    /**
     * @return the unUsedPoolItem
     */
    protected int getUnUsedPoolItem() {
        return availablePoolItemList.size();
    }

    /**
     * @return the waiterCount
     */
    protected int getWaiterCount() {
        return waiterCount;
    }

    /**
     * @param waiterCount the waiterCount to set
     */
    protected void setWaiterCount(final int waiterCount) {
        this.waiterCount = waiterCount;
    }

    /**
     * @return the interruptingAllWaiters
     */
    protected boolean isInterruptingAllWaiters() {
        return interruptingAllWaiters;
    }

    /**
     * @return the lock
     */
    protected final Lock getLock() {
        return lock;
    }

    /**
     * @return the signalClearableCondition
     */
    protected ISignalClearableCondition getSignalClearableCondition() {
        return signalClearableCondition;
    }

    /**
     * @return the delayedCount
     */
    protected int getDelayedCount() {
        return delayedCount;
    }

    /**
     * @param delayedCount the delayedCount to set
     */
    protected void setDelayedCount(final int delayedCount) {
        this.delayedCount = delayedCount;
    }

    public void interruptAllWaiters() {
        lock.lock();
        try {
            if (LOG.isDebugEnabled()) {
                LOG.debug("begin interrupt {0}/{1} (used/total)", usedPoolItem, availablePoolItemList.size() + usedPoolItem);
            }
            interruptingAllWaiters = true;
            signalClearableCondition.signalAll();
            while (waiterCount != 0) {
                try {
                    signalClearableCondition.await();
                } catch (InterruptedException e) {
                    LOG.debug("wait interrupted", e);
                }
            }
            interruptingAllWaiters = false;
            if (LOG.isDebugEnabled()) {
                LOG.debug("end interrupt {0}/{1} (used/total)", usedPoolItem, availablePoolItemList.size() + usedPoolItem);
            }
        } finally {
            lock.unlock();
        }
    }

    public void remove(final E poolItem) throws NotABusyPoolItemException {
        if (poolItem == null) {
            throw new NotABusyPoolItemException();
        }
        boolean scheduleCreation = false;

        synchronized (sizeMutex) {
            int instanceToCreate = 0;

            lock.lock();
            try {
                if (usedPoolItem <= 0) {
                    throw new NoBusyPoolItemException();
                }
                accessManager.removePoolItem(poolItem);
                usedPoolItem--;
                // calcul instance to create (or delete)
                int unUsedPoolItem = availablePoolItemList.size();
                int total = usedPoolItem + unUsedPoolItem;
                instanceToCreate = expectedSize - total;
                if (instanceToCreate < 0 && unUsedPoolItem != 0) {
                    LOG.error("issue in setExpectedSize, remove method should not make this stuff");
                }
                if (LOG.isDebugEnabled()) {
                    LOG.debug("pool {0}/{1} (used/total)", usedPoolItem, availablePoolItemList.size() + usedPoolItem);
                }
            } finally {
                lock.unlock();
            }

            // we have one less instance to create per factory working
            instanceToCreate -= scheduledCreateItemCount;
            if (instanceToCreate > 0) {
                scheduledCreateItemCount++;
                scheduleCreation = true;
                if (instanceToCreate != 1) {
                    LOG.error("issue in setExpectedSize, remove method should not schedule more than one instance creation");
                }
            }
        }
        if (scheduleCreation) {
            createItemActionScheduler.scheduleAction(1);
        }
    }

}
