/**
 * OW2 Util
 * Copyright (C) 2008 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 * --------------------------------------------------------------------------
 * $Id: BasicCluePool.java 4809 2009-03-15 10:13:49Z gaellalire $
 * --------------------------------------------------------------------------
 */

package org.ow2.util.pool.impl.enhanced.impl.basic.clue;

import java.util.List;
import java.util.concurrent.Executor;
import java.util.concurrent.locks.Lock;

import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;
import org.ow2.util.pool.impl.enhanced.api.IPoolItemFactory;
import org.ow2.util.pool.impl.enhanced.api.IllegalTimeoutException;
import org.ow2.util.pool.impl.enhanced.api.TimeoutPoolException;
import org.ow2.util.pool.impl.enhanced.api.WaiterInterruptedException;
import org.ow2.util.pool.impl.enhanced.api.basic.accessmanager.IAccessManager;
import org.ow2.util.pool.impl.enhanced.api.basic.clue.IBasicCluePool;
import org.ow2.util.pool.impl.enhanced.api.basic.clue.accessmanager.IClueAccessManager;
import org.ow2.util.pool.impl.enhanced.impl.basic.BasicPool;
import org.ow2.util.pool.impl.enhanced.internal.lock.api.ISignalClearableCondition;


/**
 * @author Gael Lalire
 * @param <E> type of pool item
 * @param <C> type of clue
 */
public class BasicCluePool<E, C> extends BasicPool<E> implements IBasicCluePool<E, C> {

    private static final Log LOG = LogFactory.getLog(BasicCluePool.class);

    private List<E> availablePoolItemList;

    private List<E> unmodifiableAvailablePoolItemList;

    private IClueAccessManager<? super E, ? super C> clueAccessManager;

    private Lock lock;

    private ISignalClearableCondition signalClearableCondition;

    public BasicCluePool(final IPoolItemFactory<? extends E> poolItemFactory, final int initPoolSize,
            final IClueAccessManager<? super E, ? super C> clueAccessManager, final Executor createItemExecutor) {
        super(poolItemFactory, initPoolSize, clueAccessManager, createItemExecutor);
        this.clueAccessManager = clueAccessManager;
        availablePoolItemList = getAvailablePoolItemList();
        unmodifiableAvailablePoolItemList = getUnmodifiableAvailablePoolItemList();
        lock = getLock();
        signalClearableCondition = getSignalClearableCondition();
    }

    /**
     * Get behavior is not the same.
     * @throws WaiterInterruptedException
     */
    @Override
    public E get(final long timeout) throws TimeoutPoolException, IllegalTimeoutException, WaiterInterruptedException {
        return get(null, timeout);
    }

    public E get(final C clue) throws WaiterInterruptedException {
        try {
            return get(clue, INFINITE_TIMEOUT);
        } catch (TimeoutPoolException e) {
            throw new Error("get(clue, INFINITE_TIMEOUT) should not return a TimeoutPoolException");
        } catch (IllegalTimeoutException e) {
            throw new Error("get(clue, INFINITE_TIMEOUT) should not return a IllegalTimeoutException");
        }
    }

    public E get(final C clue, final long timeout) throws TimeoutPoolException, IllegalTimeoutException, WaiterInterruptedException {
        if (timeout < 0 && timeout != INFINITE_TIMEOUT) {
            throw new IllegalTimeoutException();
        }
        E poolItem;
        boolean delayed = false;
        lock.lock();
        try {
            int size = availablePoolItemList.size();
            int poolItemToGetChoosed = IAccessManager.DELAY_GET;
            if (size != 0) {
                poolItemToGetChoosed = clueAccessManager.choosePoolItemToGet(unmodifiableAvailablePoolItemList, clue);
                if (poolItemToGetChoosed == IAccessManager.DELAY_GET) {
                    setDelayedCount(getDelayedCount() + 1);
                    delayed = true;
                }
            }
            if (poolItemToGetChoosed == IAccessManager.DELAY_GET) {
                long expireTime = 0;
                long timeToWait;
                if (timeout == INFINITE_TIMEOUT) {
                    timeToWait = 0;
                } else {
                    long currentTime = System.currentTimeMillis();
                    expireTime = currentTime + timeout;
                    timeToWait = expireTime - currentTime;
                    if (timeToWait <= 0) {
                        throw new TimeoutPoolException();
                    }
                }
                while (poolItemToGetChoosed == IAccessManager.DELAY_GET) {
                    if (isInterruptingAllWaiters()) {
                        throw new WaiterInterruptedException();
                    }
                    int waiterCount = getWaiterCount() + 1;
                    setWaiterCount(waiterCount);
                    LOG.debug("nb waiter increase to {0}", waiterCount);
                    try {
                        signalClearableCondition.await(timeToWait);
                        LOG.debug("wait notified or expired");
                    } catch (InterruptedException e) {
                        LOG.debug("wait interrupted");
                    }
                    waiterCount = getWaiterCount() - 1;
                    setWaiterCount(waiterCount);
                    LOG.debug("nb waiter decrease to {0}", waiterCount);
                    if (isInterruptingAllWaiters() && waiterCount == 0) {
                        signalClearableCondition.signal();
                    }
                    size = availablePoolItemList.size();
                    if (size != 0) {
                        poolItemToGetChoosed = clueAccessManager.choosePoolItemToGet(unmodifiableAvailablePoolItemList, clue);
                        if (!delayed && poolItemToGetChoosed == IAccessManager.DELAY_GET) {
                            setDelayedCount(getDelayedCount() + 1);
                            delayed = true;
                        }
                    } else {
                        if (delayed) {
                            // presume not delayed if no item available
                            setDelayedCount(getDelayedCount() - 1);
                            delayed = false;
                        }
                        if (timeout != INFINITE_TIMEOUT) {
                            timeToWait = expireTime - System.currentTimeMillis();
                            if (timeToWait <= 0) {
                                throw new TimeoutPoolException();
                            }
                        }
                    }
                }
            }
            int usedPoolItem = getUsedPoolItem();
            usedPoolItem++;
            setUsedPoolItem(usedPoolItem);
            poolItem = availablePoolItemList.remove(poolItemToGetChoosed);
            if (LOG.isDebugEnabled()) {
                LOG.debug("pool {0}/{1} (used/total)", usedPoolItem, availablePoolItemList.size() + usedPoolItem);
            }
        } finally {
            if (delayed) {
                setDelayedCount(getDelayedCount() - 1);
            }
            lock.unlock();
        }
        return poolItem;
    }

}
