/**
 * OW2 Util
 * Copyright (C) 2008 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 * --------------------------------------------------------------------------
 * $Id: KeepBusyPool.java 4759 2009-03-08 19:05:13Z gaellalire $
 * --------------------------------------------------------------------------
 */

package org.ow2.util.pool.impl.enhanced.impl.keepbusy;

import java.util.Iterator;
import java.util.Map;
import java.util.WeakHashMap;
import java.util.concurrent.locks.Lock;

import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;
import org.ow2.util.pool.impl.enhanced.api.IPool;
import org.ow2.util.pool.impl.enhanced.api.NotABusyPoolItemException;
import org.ow2.util.pool.impl.enhanced.api.PoolException;
import org.ow2.util.pool.impl.enhanced.api.TimeoutPoolException;
import org.ow2.util.pool.impl.enhanced.api.WaiterInterruptedException;
import org.ow2.util.pool.impl.enhanced.api.keepbusy.ShareMethod;
import org.ow2.util.pool.impl.enhanced.impl.util.LockFactory;

/**
 *
 * @author Gael Lalire
 * @param <E>
 */
public class KeepBusyPool<E> implements IPool<E> {

    /**
     * We keep number of user.
     */
    private Map<E, PoolItemInfo> busyPoolItemList;

    private ShareMethod shareMethod;

    private boolean interruptingAllWaiters;

    private Lock lock;

    private IPool<E> pool;

    public KeepBusyPool(final IPool<E> pool) {
        if (pool == null) {
            throw new IllegalArgumentException();
        }
        lock = LockFactory.createLock();
        this.pool = pool;
        busyPoolItemList = new WeakHashMap<E, PoolItemInfo>();
        shareMethod = ShareMethod.NEVER_SHARE;
        interruptingAllWaiters = false;
    }

    private static final Log LOG = LogFactory.getLog(KeepBusyPool.class);

    private E lookForAvailableInstance() throws PoolException {
        try {
            E poolItem = pool.get(0); // no wait with this call
            // if we add a new pool item then we have to
            // notify other waiters
            if (shareMethod != ShareMethod.NEVER_SHARE) {
                pool.interruptAllWaiters();
            }
            LOG.debug("clue pool returns NEW {0}", poolItem);
            return poolItem;
        } catch (TimeoutPoolException e) {
            return null;
        }
    }

    // FIXME: random ? less shared ? delegate choice ?
    private E lookForSharedInstance() {
        Iterator<E> iterator = busyPoolItemList.keySet().iterator();
        while (iterator.hasNext()) {
            E poolItem = iterator.next();
            LOG.debug("pool returns SHARED {0}", poolItem);
            return poolItem;
        }
        return null;
    }

    private E lookForInstance() throws PoolException {
        E poolItem;
        switch (shareMethod) {
        default:
        case NEVER_SHARE:
            poolItem = lookForAvailableInstance();
            break;
        case SHARE_AS_LAST_SOLUTION:
            poolItem = lookForAvailableInstance();
            if (poolItem == null) {
                poolItem = lookForSharedInstance();
            }
            break;
        case SHARE_IF_POSSIBLE:
            poolItem = lookForSharedInstance();
            if (poolItem == null) {
                poolItem = lookForAvailableInstance();
            }
            break;
        }
        return poolItem;
    }

    public E get(final long timeout) throws PoolException {
        if (timeout < 0 && timeout != INFINITE_TIMEOUT) {
            throw new IllegalArgumentException();
        }
        E poolItem = null;
        // take sync
        lock.lock();
        try {
            poolItem = lookForInstance();
            if (poolItem == null) {
                long expireTime = 0;
                long timeToWait;
                boolean stopWaiting = false;
                if (timeout == INFINITE_TIMEOUT) {
                    timeToWait = 0;
                } else {
                    long currentTime = System.currentTimeMillis();
                    expireTime = currentTime + timeout;
                    timeToWait = expireTime - currentTime;
                    if (timeToWait <= 0) {
                        throw new TimeoutPoolException();
                    }
                }
                while (!stopWaiting) {
                    if (interruptingAllWaiters) {
                        throw new WaiterInterruptedException();
                    }
                    try {
                        // release sync
                        lock.unlock();
                        try {
                            poolItem = pool.get(timeToWait);
                        } finally {
                            // retake sync
                            lock.lock();
                        }
                        pool.interruptAllWaiters();
                    } catch (WaiterInterruptedException e) {
                        if (interruptingAllWaiters) {
                            throw new WaiterInterruptedException();
                        }
                        if (shareMethod != ShareMethod.NEVER_SHARE) {
                            poolItem = lookForSharedInstance();
                        }
                    } catch (TimeoutPoolException e) {
                        // we could suppress next line
                        if (shareMethod != ShareMethod.NEVER_SHARE) {
                            poolItem = lookForSharedInstance();
                        }
                    }
                    if (poolItem != null) {
                        stopWaiting = true;
                    } else {
                        if (timeout != INFINITE_TIMEOUT) {
                            timeToWait = expireTime - System.currentTimeMillis();
                            if (timeToWait <= 0) {
                                throw new TimeoutPoolException();
                            }
                        }
                    }
                }
            }
            // here pool item is not null
            PoolItemInfo poolItemInfo = busyPoolItemList.get(poolItem);
            if (poolItemInfo == null) {
                busyPoolItemList.put(poolItem, new PoolItemInfo());
            } else {
                poolItemInfo.setBusyCount(poolItemInfo.getBusyCount() + 1);
            }
        } finally {
            // release sync
            lock.unlock();
        }
        // here we have no lock
        return poolItem;
    }

    /**
     * @param poolItem
     * @throws PoolException
     */
    public void put(final E poolItem) throws NotABusyPoolItemException {
        if (poolItem == null) {
            throw new IllegalArgumentException();
        }

        lock.lock();
        try {
            PoolItemInfo poolItemInfo = busyPoolItemList.get(poolItem);
            if (poolItemInfo == null) {
                throw new NotABusyPoolItemException();
            }
            int sharedCountValue = poolItemInfo.getBusyCount();
            if (sharedCountValue == 1) {
                busyPoolItemList.remove(poolItem);
                if (!poolItemInfo.isRemoved()) {
                    lock.unlock();
                    try {
                        pool.put(poolItem);
                    } finally {
                        lock.lock();
                    }
                }
            } else {
                poolItemInfo.setBusyCount(sharedCountValue - 1);
            }
            if (shareMethod != ShareMethod.NEVER_SHARE) {
                pool.interruptAllWaiters();
            }
        } finally {
            lock.unlock();
        }

    }

    protected ShareMethod getShareMethod0() {
        return shareMethod;
    }


    /**
     * @return the shareMethod
     */
    public ShareMethod getShareMethod() {
        lock.lock();
        try {
            return shareMethod;
        } finally {
            lock.unlock();
        }
    }

    /**
     * @param shareMethod the shareMethod to set
     */
    public void setShareMethod(final ShareMethod shareMethod) {
        lock.lock();
        try {
            this.shareMethod = shareMethod;
        } finally {
            lock.unlock();
        }
    }

    public void interruptAllWaiters() {
        lock.lock();

        try {
            interruptingAllWaiters = true;
        } finally {
            lock.unlock();
        }
        pool.interruptAllWaiters();
        lock.lock();
        try {
            interruptingAllWaiters = false;
        } finally {
            lock.unlock();
        }
    }

    public void remove(final E poolItem) throws NotABusyPoolItemException {
        if (poolItem == null) {
            throw new IllegalArgumentException();
        }
        lock.lock();

        try {
            PoolItemInfo poolItemInfo = busyPoolItemList.get(poolItem);
            if (poolItemInfo == null) {
                throw new NotABusyPoolItemException();
            }
            if (!poolItemInfo.isRemoved()) {
                lock.unlock();
                try {
                    pool.remove(poolItem);
                } finally {
                    lock.lock();
                }
                poolItemInfo.setRemoved(true);
            }
            int sharedCountValue = poolItemInfo.getBusyCount();
            if (sharedCountValue == 1) {
                busyPoolItemList.remove(poolItem);
            } else {
                poolItemInfo.setBusyCount(sharedCountValue - 1);
            }
        } finally {
            lock.unlock();
        }
    }

    /**
     * @return the interruptingAllWaiters
     */
    protected boolean isInterruptingAllWaiters() {
        return interruptingAllWaiters;
    }

    /**
     * @return the busyPoolItemList
     */
    protected Map<E, PoolItemInfo> getBusyPoolItemList() {
        return busyPoolItemList;
    }

    /**
     * @return the lock
     */
    protected Lock getLock() {
        return lock;
    }

}
