/**
 * OW2 Util
 * Copyright (C) 2008 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 * --------------------------------------------------------------------------
 * $Id: KeepBusyCluePool.java 4759 2009-03-08 19:05:13Z gaellalire $
 * --------------------------------------------------------------------------
 */

package org.ow2.util.pool.impl.enhanced.impl.keepbusy.clue;

import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.locks.Lock;

import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;
import org.ow2.util.pool.impl.enhanced.api.PoolException;
import org.ow2.util.pool.impl.enhanced.api.TimeoutPoolException;
import org.ow2.util.pool.impl.enhanced.api.WaiterInterruptedException;
import org.ow2.util.pool.impl.enhanced.api.clue.IClueManager;
import org.ow2.util.pool.impl.enhanced.api.clue.ICluePool;
import org.ow2.util.pool.impl.enhanced.api.keepbusy.ShareMethod;
import org.ow2.util.pool.impl.enhanced.impl.keepbusy.KeepBusyPool;
import org.ow2.util.pool.impl.enhanced.impl.keepbusy.PoolItemInfo;


/**
 * @author Gael Lalire
 * @param <E> type of pool item
 * @param <C> type of clue
 */
public class KeepBusyCluePool<E, C> extends KeepBusyPool<E> implements ICluePool<E, C> {

    private static final Log LOG = LogFactory.getLog(KeepBusyCluePool.class);

    /**
     * We keep number of user.
     */
    private Map<E, PoolItemInfo> busyPoolItemList;

    private IClueManager<E, C> clueManager;

    private ICluePool<E, C> cluePool;

    private Lock lock;

    public KeepBusyCluePool(final ICluePool<E, C> cluePool, final IClueManager<E, C> clueManager) {
        super(cluePool);
        if (clueManager == null) {
            throw new IllegalArgumentException();
        }
        busyPoolItemList = getBusyPoolItemList();
        this.cluePool = cluePool;
        this.clueManager = clueManager;
        lock = getLock();
    }

    /**
     * Because not same behavior.
     */
    @Override
    public E get(final long timeout) throws PoolException {
        return get(null, timeout);
    }

    private E lookForAvailableInstance(final C clue) throws PoolException {
        try {
            E poolItem = cluePool.get(clue, 0); // no wait with this call
            // if we add a new pool item then we have to
            // notify other waiters
            if (getShareMethod0() != ShareMethod.NEVER_SHARE) {
                cluePool.interruptAllWaiters();
            }
            LOG.debug("clue pool returns NEW {0}", poolItem);
            return poolItem;
        } catch (TimeoutPoolException e) {
            return null;
        }
    }

    private E lookForSharedInstance(final C clue) {
        Iterator<E> iterator = busyPoolItemList.keySet().iterator();
        while (iterator.hasNext()) {
            E poolItem = iterator.next();
            if (clueManager.tryMatch(poolItem, clue)) {
                LOG.debug("clue pool returns SHARED {0}", poolItem);
                return poolItem;
            }
        }
        return null;
    }

    private E lookForInstance(final C clue) throws PoolException {
        E poolItem;
        switch (getShareMethod0()) {
        default:
        case NEVER_SHARE:
            poolItem = lookForAvailableInstance(clue);
            break;
        case SHARE_AS_LAST_SOLUTION:
            poolItem = lookForAvailableInstance(clue);
            if (poolItem == null) {
                poolItem = lookForSharedInstance(clue);
            }
            break;
        case SHARE_IF_POSSIBLE:
            poolItem = lookForSharedInstance(clue);
            if (poolItem == null) {
                poolItem = lookForAvailableInstance(clue);
            }
            break;
        }
        return poolItem;
    }

    public E get(final C clue, final long timeout) throws PoolException {
        if (timeout < 0 && timeout != INFINITE_TIMEOUT) {
            throw new IllegalArgumentException();
        }
        E poolItem = null;
        // take sync
        lock.lock();
        try {
            poolItem = lookForInstance(clue);
            if (poolItem == null) {
                long expireTime = 0;
                long timeToWait;
                boolean stopWaiting = false;
                if (timeout == INFINITE_TIMEOUT) {
                    timeToWait = 0;
                } else {
                    long currentTime = System.currentTimeMillis();
                    expireTime = currentTime + timeout;
                    timeToWait = expireTime - currentTime;
                    if (timeToWait <= 0) {
                        throw new TimeoutPoolException();
                    }
                }
                while (!stopWaiting) {
                    if (isInterruptingAllWaiters()) {
                        throw new WaiterInterruptedException();
                    }
                    try {
                        // release sync
                        lock.unlock();
                        try {
                            poolItem = cluePool.get(clue, timeToWait);
                        } finally {
                            // retake sync
                            lock.lock();
                        }
                        cluePool.interruptAllWaiters();
                    } catch (WaiterInterruptedException e) {
                        if (isInterruptingAllWaiters()) {
                            throw new WaiterInterruptedException();
                        }
                        if (getShareMethod0() != ShareMethod.NEVER_SHARE) {
                            poolItem = lookForSharedInstance(clue);
                        }
                    } catch (TimeoutPoolException e) {
                        // we could suppress next line
                        if (getShareMethod0() != ShareMethod.NEVER_SHARE) {
                            poolItem = lookForSharedInstance(clue);
                        }
                    }
                    if (poolItem != null) {
                        stopWaiting = true;
                    } else {
                        if (timeout != INFINITE_TIMEOUT) {
                            timeToWait = expireTime - System.currentTimeMillis();
                            if (timeToWait <= 0) {
                                throw new TimeoutPoolException();
                            }
                        }
                    }
                }
            }
            // here pool item is not null
            PoolItemInfo poolItemInfo = busyPoolItemList.get(poolItem);
            if (poolItemInfo == null) {
                busyPoolItemList.put(poolItem, new PoolItemInfo());
            } else {
                poolItemInfo.setBusyCount(poolItemInfo.getBusyCount() + 1);
            }
        } finally {
            // release sync
            lock.unlock();
        }
        // here we have no lock
        return poolItem;
    }

}
