/**
 * OW2 Util
 * Copyright (C) 2008 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 * --------------------------------------------------------------------------
 * $Id: LimitedWaiterPool.java 4389 2008-12-15 13:48:57Z alitokmen $
 * --------------------------------------------------------------------------
 */

package org.ow2.util.pool.impl.enhanced.impl.limited;

import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;
import org.ow2.util.pool.impl.enhanced.EnhancedPool;
import org.ow2.util.pool.impl.enhanced.api.IPool;
import org.ow2.util.pool.impl.enhanced.api.NotABusyPoolItemException;
import org.ow2.util.pool.impl.enhanced.api.PoolException;


/**
 *
 * @author Gael Lalire
 * @param <E>
 */
public class LimitedWaiterPool<E> implements IPool<E> {

    public static final int NO_LIMIT_WAITER = -1;

    private static final Log LOG = LogFactory.getLog(EnhancedPool.class);

    private Object mutex;

    private int maxWaiter;

    private int currentWaiter;

    private IPool<E> pool;

    public LimitedWaiterPool(final IPool<E> pool) {
        this(pool, NO_LIMIT_WAITER);
    }

    public LimitedWaiterPool(final IPool<E> pool, final int maxWaiter) {
        if (maxWaiter < 0 && maxWaiter != NO_LIMIT_WAITER) {
            throw new IllegalArgumentException();
        }
        mutex = new Object();
        this.pool = pool;
        this.maxWaiter = maxWaiter;
    }

    public E get(final long timeout) throws PoolException {
        if (timeout == 0) {
            return pool.get(0); // return no wait
        } else {
            addWaiter();
            try {
                return pool.get(timeout);
            } finally {
                removeWaiter();
            }
        }
    }

    protected void addWaiter() throws TooManyWaiterException {
        synchronized (mutex) {
            if (maxWaiter != NO_LIMIT_WAITER && currentWaiter >= maxWaiter) {
                throw new TooManyWaiterException();
            }
            currentWaiter++;
            LOG.debug("there is {0} waiters ", currentWaiter);
        }
    }

    protected void removeWaiter() throws TooManyWaiterException {
        synchronized (mutex) {
            currentWaiter--;
            LOG.debug("there is {0} waiters ", currentWaiter);
        }
    }

    public void put(final E e) throws NotABusyPoolItemException {
        pool.put(e);
    }

    /**
     * @param maxWaiter the maxWaiter to set
     */
    public void setMaxWaiter(final int maxWaiter) {
        if (maxWaiter < 0) {
            throw new IllegalArgumentException();
        }
        synchronized (mutex) {
            this.maxWaiter = maxWaiter;
        }
    }

    public void interruptAllWaiters() {
        pool.interruptAllWaiters();
    }

    public void remove(final E poolItem) throws NotABusyPoolItemException {
        pool.remove(poolItem);
    }

}
