/**
 * OW2 Util
 * Copyright (C) 2008 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 * --------------------------------------------------------------------------
 * $Id: LimitedWaiterCluePool.java 4389 2008-12-15 13:48:57Z alitokmen $
 * --------------------------------------------------------------------------
 */

package org.ow2.util.pool.impl.enhanced.impl.limited.clue;

import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;
import org.ow2.util.pool.impl.enhanced.EnhancedPool;
import org.ow2.util.pool.impl.enhanced.api.PoolException;
import org.ow2.util.pool.impl.enhanced.api.clue.ICluePool;
import org.ow2.util.pool.impl.enhanced.impl.limited.LimitedWaiterPool;
import org.ow2.util.pool.impl.enhanced.impl.limited.TooManyWaiterException;


/**
 *
 * @author Gael Lalire
 * @param <E>
 * @param <C>
 */
public class LimitedWaiterCluePool<E, C> extends LimitedWaiterPool<E> implements ICluePool<E, C> {

    private static final Log LOG = LogFactory.getLog(EnhancedPool.class);

    private Object mutex;

    private int maxClueWaiter;

    private int currentClueWaiter;

    private int maxNoClueWaiter;

    private int currentNoClueWaiter;

    private ICluePool<E, C> cluePool;

    public LimitedWaiterCluePool(final ICluePool<E, C> cluePool) {
        this(cluePool, NO_LIMIT_WAITER, NO_LIMIT_WAITER, NO_LIMIT_WAITER);
    }

    public LimitedWaiterCluePool(final ICluePool<E, C> cluePool, final int maxWaiter) {
        this(cluePool, maxWaiter, NO_LIMIT_WAITER, NO_LIMIT_WAITER);
    }

    public LimitedWaiterCluePool(final ICluePool<E, C> cluePool, final int maxWaiter, final int maxClueWaiter,
            final int maxNoClueWaiter) {
        super(cluePool, maxWaiter);
        if ((maxClueWaiter < 0 && maxClueWaiter != NO_LIMIT_WAITER)
                || (maxNoClueWaiter < 0 && maxNoClueWaiter != NO_LIMIT_WAITER)) {
            throw new IllegalArgumentException();
        }
        mutex = new Object();
        this.maxClueWaiter = maxClueWaiter;
        this.maxNoClueWaiter = maxNoClueWaiter;
        this.cluePool = cluePool;
    }

    public E get(final C clue) throws PoolException {
        return get(clue, INFINITE_TIMEOUT);
    }

    @Override
    public E get(final long timeout) throws PoolException {
        if (timeout == 0) {
            return cluePool.get(0); // return no wait
        } else {
            addWaiter();
            addNoClueWaiter();
            try {
                return cluePool.get(timeout);
            } finally {
                removeNoClueWaiter();
                removeWaiter();
            }
        }
    }

    protected void addClueWaiter() throws TooManyWaiterException {
        synchronized (mutex) {
            if (maxClueWaiter != NO_LIMIT_WAITER && currentClueWaiter >= maxClueWaiter) {
                removeWaiter();
                throw new TooManyWaiterException();
            }
            currentClueWaiter++;
        }
    }

    protected void removeClueWaiter() throws TooManyWaiterException {
        synchronized (mutex) {
            currentClueWaiter--;
        }
    }

    protected void addNoClueWaiter() throws TooManyWaiterException {
        synchronized (mutex) {
            if (maxNoClueWaiter != NO_LIMIT_WAITER && currentNoClueWaiter >= maxNoClueWaiter) {
                removeWaiter();
                throw new TooManyWaiterException();
            }
            currentNoClueWaiter++;
        }
    }

    protected void removeNoClueWaiter() throws TooManyWaiterException {
        synchronized (mutex) {
            currentNoClueWaiter--;
        }
    }

    public E get(final C clue, final long timeout) throws PoolException {
        if (timeout == 0) {
            return cluePool.get(clue, 0); // return no wait
        } else {
            addWaiter();
            if (clue == null) {
                addNoClueWaiter();
            } else {
                addClueWaiter();
            }
            try {
                return cluePool.get(clue, timeout);
            } finally {
                if (clue == null) {
                    removeNoClueWaiter();
                } else {
                    removeClueWaiter();
                }
                removeWaiter();
            }
        }
    }

}
