/**
 * OW2 Util
 * Copyright (C) 2008 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 * --------------------------------------------------------------------------
 * $Id: PoolListenerDispatcherPool.java 4389 2008-12-15 13:48:57Z alitokmen $
 * --------------------------------------------------------------------------
 */

package org.ow2.util.pool.impl.enhanced.impl.listener;

import java.util.ArrayList;
import java.util.List;

import org.ow2.util.pool.impl.enhanced.api.IPool;
import org.ow2.util.pool.impl.enhanced.api.NotABusyPoolItemException;
import org.ow2.util.pool.impl.enhanced.api.PoolException;
import org.ow2.util.pool.impl.enhanced.api.listener.IPoolListener;

/**
 *
 * @author Gael Lalire
 * @param <E>
 */
public class PoolListenerDispatcherPool<E> implements IPool<E> {

    private IPool<E> pool;

    private List<IPoolListener<E>> poolListenerList;

    public PoolListenerDispatcherPool(final IPool<E> pool) {
        this.pool = pool;
        poolListenerList = new ArrayList<IPoolListener<E>>();
    }

    public E get(final long timeout) throws PoolException {
        for (IPoolListener<E> poolListener : poolListenerList) {
            poolListener.getMethodCalled(timeout);
        }
        try {
            E result = pool.get(timeout);
            for (IPoolListener<E> poolListener : poolListenerList) {
                poolListener.getMethodReturned(result, timeout);
            }
            return result;
        } catch (PoolException poolException) {
            for (IPoolListener<E> poolListener : poolListenerList) {
                poolListener.getMethodFailed(poolException, timeout);
            }
            throw poolException;
        }
    }

    public void interruptAllWaiters() {
        for (IPoolListener<E> poolListener : poolListenerList) {
            poolListener.interruptAllWaitersMethodCalled();
        }
        pool.interruptAllWaiters();
        for (IPoolListener<E> poolListener : poolListenerList) {
            poolListener.interruptAllWaitersMethodReturned();
        }
    }

    public void put(final E poolItem) throws NotABusyPoolItemException {
        for (IPoolListener<E> poolListener : poolListenerList) {
            poolListener.putMethodCalled(poolItem);
        }
        try {
            pool.put(poolItem);
            for (IPoolListener<E> poolListener : poolListenerList) {
                poolListener.putMethodReturned(poolItem);
            }
        } catch (NotABusyPoolItemException poolException) {
            for (IPoolListener<E> poolListener : poolListenerList) {
                poolListener.putMethodFailed(poolException, poolItem);
            }
            throw poolException;
        }
    }

    public void remove(final E poolItem) throws NotABusyPoolItemException {
        for (IPoolListener<E> poolListener : poolListenerList) {
            poolListener.removeMethodCalled(poolItem);
        }
        try {
            pool.put(poolItem);
            for (IPoolListener<E> poolListener : poolListenerList) {
                poolListener.removeMethodReturned(poolItem);
            }
        } catch (NotABusyPoolItemException poolException) {
            for (IPoolListener<E> poolListener : poolListenerList) {
                poolListener.removeMethodFailed(poolException, poolItem);
            }
            throw poolException;
        }
    }

    public void addPoolListener(final IPoolListener<E> poolListener) {
        poolListenerList.add(poolListener);
    }

    public void removePoolListener(final IPoolListener<E> poolListener) {
        poolListenerList.remove(poolListener);
    }

}
