/**
 * OW2 Util
 * Copyright (C) 2008 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 * --------------------------------------------------------------------------
 * $Id: ResizerUpdatePoolListener.java 4389 2008-12-15 13:48:57Z alitokmen $
 * --------------------------------------------------------------------------
 */

package org.ow2.util.pool.impl.enhanced.impl.listener.resizer;

import org.ow2.util.pool.impl.enhanced.api.PoolException;
import org.ow2.util.pool.impl.enhanced.api.listener.IPoolListener;
import org.ow2.util.pool.impl.enhanced.api.listener.statistics.IPoolStatsListener;
import org.ow2.util.pool.impl.enhanced.internal.resizer.api.IResizer;

/**
 *
 * @author Gael Lalire
 * @param <E>
 * @param <S>
 */
public class ResizerUpdatePoolListener<E, S> implements IPoolListener<E> {

    private UpdateMethod getUpdateMethod;

    private UpdateMethod putUpdateMethod;

    private UpdateMethod removeUpdateMethod;

    private UpdateMethod interruptAllWaitersUpdateMethod;

    private IPoolStatsListener<E, S> poolStatsAccessor;

    private IResizer<S> poolResizer;

    private Object mutex;

    private S currentStats;

    public ResizerUpdatePoolListener(final S stats, final IResizer<S> poolResizer,
            final IPoolStatsListener<E, S> poolStatsAccessor) {
        this(stats, poolResizer, poolStatsAccessor, UpdateMethod.BEFORE, UpdateMethod.AFTER, UpdateMethod.AFTER,
                UpdateMethod.AFTER);
    }

    public ResizerUpdatePoolListener(final S stats, final IResizer<S> poolResizer,
            final IPoolStatsListener<E, S> poolStatsAccessor, final UpdateMethod getUpdateMethod,
            final UpdateMethod putUpdateMethod, final UpdateMethod removeUpdateMethod,
            final UpdateMethod interruptAllWaitersUpdateMethod) {
        if (stats == null) {
            throw new IllegalArgumentException();
        }
        this.poolStatsAccessor = poolStatsAccessor;
        this.poolResizer = poolResizer;
        this.currentStats = stats;
        this.getUpdateMethod = getUpdateMethod;
        this.putUpdateMethod = putUpdateMethod;
        this.removeUpdateMethod = removeUpdateMethod;
        this.interruptAllWaitersUpdateMethod = interruptAllWaitersUpdateMethod;
        mutex = new Object();
        // update on creation
        prepareUpdate();
        poolResizer.update();
    }

    protected void prepareUpdate() {
        synchronized (mutex) {
            poolStatsAccessor.copyStats(currentStats);
            poolResizer.prepareUpdate(currentStats);
        }
    }

    public void getMethodCalled(final long timeout) {
        prepareUpdate();
        if (getUpdateMethod.isBefore()) {
            poolResizer.update();
        }
    }

    public void getMethodFailed(final PoolException poolException, final long timeout) {
        prepareUpdate();
        if (getUpdateMethod.isAfterFailed()) {
            poolResizer.update();
        }
    }

    public void getMethodReturned(final E result, final long timeout) {
        prepareUpdate();
        if (getUpdateMethod.isAfterSuccessful()) {
            poolResizer.update();
        }
    }

    public void interruptAllWaitersMethodCalled() {
        prepareUpdate();
        if (interruptAllWaitersUpdateMethod.isBefore()) {
            poolResizer.update();
        }
    }

    public void interruptAllWaitersMethodReturned() {
        prepareUpdate();
        if (interruptAllWaitersUpdateMethod.isAfterSuccessful()) {
            poolResizer.update();
        }
    }

    public void putMethodCalled(final E poolItem) {
        prepareUpdate();
        if (putUpdateMethod.isBefore()) {
            poolResizer.update();
        }
    }

    public void putMethodFailed(final PoolException poolException, final E poolItem) {
        prepareUpdate();
        if (putUpdateMethod.isAfterFailed()) {
            poolResizer.update();
        }
    }

    public void putMethodReturned(final E poolItem) {
        prepareUpdate();
        if (putUpdateMethod.isAfterSuccessful()) {
            poolResizer.update();
        }
    }

    public void removeMethodCalled(final E poolItem) {
        prepareUpdate();
        if (removeUpdateMethod.isBefore()) {
            poolResizer.update();
        }
    }

    public void removeMethodFailed(final PoolException poolException, final E poolItem) {
        prepareUpdate();
        if (removeUpdateMethod.isAfterFailed()) {
            poolResizer.update();
        }
    }

    public void removeMethodReturned(final E poolItem) {
        prepareUpdate();
        if (removeUpdateMethod.isAfterSuccessful()) {
            poolResizer.update();
        }
    }

}
