/**
 * OW2 Util
 * Copyright (C) 2009 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 * --------------------------------------------------------------------------
 * $Id: RecorderPool.java 4809 2009-03-15 10:13:49Z gaellalire $
 * --------------------------------------------------------------------------
 */

package org.ow2.util.pool.impl.enhanced.impl.recorder;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.WeakHashMap;
import java.util.concurrent.Callable;
import java.util.concurrent.Executor;
import java.util.concurrent.locks.Lock;

import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;
import org.ow2.util.pool.impl.enhanced.api.IPoolItemFactory;
import org.ow2.util.pool.impl.enhanced.api.IllegalTimeoutException;
import org.ow2.util.pool.impl.enhanced.api.NotABusyPoolItemException;
import org.ow2.util.pool.impl.enhanced.api.TimeoutPoolException;
import org.ow2.util.pool.impl.enhanced.api.WaiterInterruptedException;
import org.ow2.util.pool.impl.enhanced.api.basic.NoBusyPoolItemException;
import org.ow2.util.pool.impl.enhanced.api.basic.accessmanager.IAccessManager;
import org.ow2.util.pool.impl.enhanced.api.recorder.IPoolItemRecorder;
import org.ow2.util.pool.impl.enhanced.api.recorder.IRecorderPool;
import org.ow2.util.pool.impl.enhanced.api.recorder.RecorderPoolItemState;
import org.ow2.util.pool.impl.enhanced.api.recorder.accesmanager.IRecorderAccessManager;
import org.ow2.util.pool.impl.enhanced.impl.util.LockFactory;
import org.ow2.util.pool.impl.enhanced.internal.actionscheduler.ActionSchedulerFactory;
import org.ow2.util.pool.impl.enhanced.internal.actionscheduler.IActionScheduler;
import org.ow2.util.pool.impl.enhanced.internal.lock.api.ISignalClearableCondition;
import org.ow2.util.pool.impl.enhanced.internal.lock.impl.SignalClearableConditionProxy;

/**
 * TODO FIXME : a createNewItem schedule may/should abort a fetch schedule (to prevent
 * a useless record) ?
 * TODO FIXME : get method if we choose a recorded item ? is it allowed (if not DELAY_GET
 * must be returned), issue with the fact that an fetch(record) is not like
 * Identity but record and fetch abort their respective opposite action ... ?
 * @author Gael Lalire
 */
public class RecorderPool<E, I> implements IRecorderPool<E> {

    private static final Log LOG = LogFactory.getLog(RecorderPool.class);

    private Lock lock;

    private ISignalClearableCondition signalClearableCondition;

    private Map<E, RecorderPoolItem<E, I>> busyPoolItem;

    private int scheduledFetchItemCount;

    private IActionScheduler fetchItemActionScheduler;

    private int scheduledRecordItemCount;

    private IActionScheduler recordItemActionScheduler;

    private Object recordMutex = new Object();

    private int usedPoolItem;

    private List<RecorderPoolItem<E, I>> availablePoolItemList;

    private List<RecorderPoolItem<E, I>> unmodifiableAvailablePoolItemList;

    private IPoolItemFactory<? extends E> poolItemFactory;

    private int expectedSize;

    private IRecorderAccessManager<? super E, ? super I> accessManager;

    private Object sizeMutex;

    private int waiterCount;

    private boolean interruptingAllWaiters;

    private int scheduledCreateItemCount;

    private IActionScheduler createItemActionScheduler;

    private IPoolItemRecorder<E, I> poolItemRecorder;

    private int delayedCount;

    public RecorderPool(final IPoolItemFactory<? extends E> poolItemFactory, final int initPoolSize,
            final IRecorderAccessManager<E, I> accessManager, final Executor createItemExecutor,
            final Executor recordItemExecutor, final Executor fetchItemExecutor) {
        if (initPoolSize < 0 || poolItemFactory == null || accessManager == null || createItemExecutor == null
                || recordItemExecutor == null || fetchItemExecutor == null) {
            throw new IllegalArgumentException();
        }
        lock = LockFactory.createLock();
        signalClearableCondition = SignalClearableConditionProxy.createProxy(lock.newCondition());
        sizeMutex = new Object();
        delayedCount = 0;
        scheduledCreateItemCount = 0;
        waiterCount = 0;
        interruptingAllWaiters = false;
        availablePoolItemList = new ArrayList<RecorderPoolItem<E, I>>();
        unmodifiableAvailablePoolItemList = Collections.unmodifiableList(availablePoolItemList);
        this.accessManager = accessManager;
        this.poolItemFactory = poolItemFactory;
        expectedSize = initPoolSize;
        for (int i = 0; i < initPoolSize; i++) {
            E poolItem = poolItemFactory.createPoolItem();
            if (poolItem == null) {
                throw new NullPointerException("poolItemFactory returns a null instance");
            }
            availablePoolItemList.add(new RecorderPoolItem<E, I>(poolItem));
        }
        usedPoolItem = 0;
        createItemActionScheduler = new ActionSchedulerFactory().createActionScheduler(new Callable<Boolean>() {

            public Boolean call() throws Exception {
                 return createNewItem();
            }

        }, createItemExecutor);
        ActionSchedulerFactory actionSchedulerFactory = new ActionSchedulerFactory();
        fetchItemActionScheduler = actionSchedulerFactory.createActionScheduler(new Callable<Boolean>() {

            public Boolean call() throws Exception {
                return fetch();
            }

        }, fetchItemExecutor);
        recordItemActionScheduler = actionSchedulerFactory.createActionScheduler(new Callable<Boolean>() {

            public Boolean call() throws Exception {
                return record();
            }

        }, recordItemExecutor);
        busyPoolItem = new WeakHashMap<E, RecorderPoolItem<E, I>>();
    }

    protected boolean record() {

        RecorderPoolItem<E, I> recorderPoolItem;
        RecorderPoolItemState previousState;

        synchronized (recordMutex) {
            lock.lock();
            try {
                // try to abort a fetch instead
                int cancelled = fetchItemActionScheduler.cancelAction(1);
                if (cancelled != 0) {
                    scheduledFetchItemCount--;
                    scheduledRecordItemCount--;
                    return false;
                }

                int poolItemToRecordChoosed = accessManager.choosePoolItemToRecord(unmodifiableAvailablePoolItemList);
                recorderPoolItem = availablePoolItemList.get(poolItemToRecordChoosed);
                previousState = recorderPoolItem.getState();
                if (!previousState.isRecordable()) {
                    LOG.error("accessManager choose an invalid item to be recorded");
                    Iterator<RecorderPoolItem<E, I>> iterator = availablePoolItemList.iterator();
                    while (iterator.hasNext()) {
                        recorderPoolItem = iterator.next();
                        previousState = recorderPoolItem.getState();
                        if (previousState.isRecordable()) {
                            break;
                        }
                    }
                    if (!previousState.isRecordable()) {
                        throw new Error("record was scheduled whereas no item can be recorded");
                    }
                }

                switch (previousState) {
                case FETCHING:
                    // cancel fetching
                    recorderPoolItem.setState(RecorderPoolItemState.ABORTED_FETCHING);
                    movePoolItemWhoseStateChanged(recorderPoolItem, previousState);
                    scheduledRecordItemCount--;
                    scheduledFetchItemCount--;
                    return false;
                case ABORTED_RECORDING:
                    recorderPoolItem.setState(RecorderPoolItemState.RECORDING);
                    movePoolItemWhoseStateChanged(recorderPoolItem, previousState);
                    return false;
                case FETCHED:
                    // normal case
                    break;
                default:
                    throw new Error();
                }
                recorderPoolItem.setState(RecorderPoolItemState.RECORDING);
                movePoolItemWhoseStateChanged(recorderPoolItem, previousState);
            } finally {
                lock.unlock();
            }
        }

        I id = poolItemRecorder.record(recorderPoolItem.getItem(), recorderPoolItem.getId());

        synchronized (recordMutex) {
            lock.lock();
            try {

                RecorderPoolItemState state = recorderPoolItem.getState();
                if (state == RecorderPoolItemState.RECORDING) {
                    recorderPoolItem.setId(id);
                    recorderPoolItem.setItem(null);
                    recorderPoolItem.setState(RecorderPoolItemState.RECORDED);
                    movePoolItemWhoseStateChanged(recorderPoolItem, previousState);
                    recordableItemSize--;
                    scheduledRecordItemCount--;
                } else if (state == RecorderPoolItemState.ABORTED_RECORDING) {
                    recorderPoolItem.setId(id);
                    recorderPoolItem.setState(RecorderPoolItemState.FETCHED);
                    movePoolItemWhoseStateChanged(recorderPoolItem, previousState);
                } else {
                    throw new Error();
                }
            } finally {
                lock.unlock();
            }
        }

        return true;
    }

    protected boolean fetch() {

        RecorderPoolItem<E, I> recorderPoolItem;
        RecorderPoolItemState previousState;

        synchronized (recordMutex) {
            lock.lock();
            try {
                // try to abort a record instead
                int cancelled = recordItemActionScheduler.cancelAction(1);
                if (cancelled != 0) {
                    scheduledRecordItemCount--;
                    scheduledFetchItemCount--;
                    return false;
                }

                int poolItemToFetchChoosed = accessManager.choosePoolItemToFetch(unmodifiableAvailablePoolItemList);
                recorderPoolItem = availablePoolItemList.get(poolItemToFetchChoosed);
                previousState = recorderPoolItem.getState();
                if (!previousState.isFetchable()) {
                    LOG.error("accessManager choose an invalid item to be fetched");
                    Iterator<RecorderPoolItem<E, I>> iterator = availablePoolItemList.iterator();
                    while (iterator.hasNext()) {
                        recorderPoolItem = iterator.next();
                        previousState = recorderPoolItem.getState();
                        if (previousState.isFetchable()) {
                            break;
                        }
                    }
                    if (!previousState.isFetchable()) {
                        throw new Error("fetch was scheduled whereas no item can be fetched");
                    }
                }
                switch (previousState) {
                case RECORDING:
                    // cancel recording
                    recorderPoolItem.setState(RecorderPoolItemState.ABORTED_RECORDING);
                    movePoolItemWhoseStateChanged(recorderPoolItem, previousState);
                    scheduledFetchItemCount--;
                    scheduledRecordItemCount--;
                    return false;
                case ABORTED_FETCHING:
                    recorderPoolItem.setState(RecorderPoolItemState.FETCHING);
                    movePoolItemWhoseStateChanged(recorderPoolItem, previousState);
                    return false;
                case RECORDED:
                    // normal case
                    break;
                default:
                    throw new Error();
                }
                recorderPoolItem.setState(RecorderPoolItemState.FETCHING);
                movePoolItemWhoseStateChanged(recorderPoolItem, previousState);
            } finally {
                lock.unlock();
            }
        }


        E item = poolItemRecorder.fetch(recorderPoolItem.getId());

        synchronized (recordMutex) {
            lock.lock();
            try {

                RecorderPoolItemState state = recorderPoolItem.getState();
                if (state == RecorderPoolItemState.FETCHING) {
                    recorderPoolItem.setItem(item);
                    recorderPoolItem.setState(RecorderPoolItemState.FETCHED);
                    movePoolItemWhoseStateChanged(recorderPoolItem, previousState);
                    recordableItemSize++;
                    scheduledFetchItemCount--;
                } else if (state == RecorderPoolItemState.ABORTED_FETCHING) {
                    recorderPoolItem.setState(RecorderPoolItemState.RECORDED);
                    movePoolItemWhoseStateChanged(recorderPoolItem, previousState);
                } else {
                    throw new Error();
                }
            } finally {
                lock.unlock();
            }
        }

        return true;
    }

    /**
     * Call this method if you own at least lock. The state of recorderPoolItem
     * have to already be modified.
     */
    private void movePoolItemWhoseStateChanged(final RecorderPoolItem<E, I> recorderPoolItem,
            final RecorderPoolItemState previousState) {
        int distance = accessManager.movePoolItemWhoseStateChanged(unmodifiableAvailablePoolItemList, recorderPoolItem, previousState);
        if (distance != 0) {
            int oldPosition = availablePoolItemList.indexOf(recorderPoolItem);
            int size = availablePoolItemList.size();
            int newPosition = (oldPosition + distance) % size; // ] -size, size [
            if (newPosition < 0) {
                newPosition += size; // [0, size [
            }
            if (newPosition > oldPosition) {
                // { 0, 1, {o, 2, 3, n}, 4, 5}
                Collections.rotate(availablePoolItemList.subList(oldPosition, newPosition + 1), -1);
                // { 0, 1, {2, 3, n, o}, 4, 5}
            } else if (newPosition != oldPosition) {
                // { 0, 1, {n, 2, 3, o}, 4, 5}
                Collections.rotate(availablePoolItemList.subList(newPosition, oldPosition + 1), 1);
                // { 0, 1, {o, n, 2, 3}, 4, 5}
            }
        }
    }

    private int recordableItemSize;

    private int expectedRecordableItemSize;

    /**
     * @param expectedRecordableItemSize the expectedRecordableItemSize to set
     */
    public void setExpectedRecordableItemSize(final int expectedRecordableItemSize) {
        if (expectedRecordableItemSize < 0) {
            throw new IllegalArgumentException();
        }
        LOG.debug("Want mutex for setting fetched size to {0}", expectedRecordableItemSize);
        int instanceToRecord = 0;

        synchronized (recordMutex) {
            lock.lock();
            try {
                if (this.expectedRecordableItemSize != expectedRecordableItemSize) {
                    LOG.debug("size will change from {0} to {1}", this.expectedRecordableItemSize, expectedRecordableItemSize);
                    this.expectedRecordableItemSize = expectedRecordableItemSize;
                }
                int futurRecordableItemSize = recordableItemSize + scheduledFetchItemCount - scheduledRecordItemCount;

                instanceToRecord = futurRecordableItemSize - expectedRecordableItemSize;
                if (instanceToRecord > 0) {
                    // too much recordable item in futur, need to record
                    instanceToRecord -= fetchItemActionScheduler.cancelAction(instanceToRecord);
                } else if (instanceToRecord != 0) {
                    // not enough recordable item in futur
                    instanceToRecord += recordItemActionScheduler.cancelAction(-instanceToRecord);
                }

            } finally {
                lock.unlock();
            }

            if (instanceToRecord > 0) {
                scheduledRecordItemCount++;
            } else if (instanceToRecord != 0) {
                scheduledFetchItemCount++;
            }
        }
        // ask to call 'instanceToCreate' times the createNewItem() method
        if (instanceToRecord > 0) {
            recordItemActionScheduler.scheduleAction(instanceToRecord);
        } else if (instanceToRecord != 0) {
            fetchItemActionScheduler.scheduleAction(-instanceToRecord);
        }

    }


    /**
     * @param expectedSize the expectedSize to set
     */
    public void setExpectedSize(final int expectedSize) {
        if (expectedSize < 0) {
            throw new IllegalArgumentException();
        }
        LOG.debug("Want mutex for setting size to {0}", expectedSize);
        int instanceToCreate = 0;

        synchronized (sizeMutex) {
            lock.lock();
            try {
                if (this.expectedSize != expectedSize) {
                    LOG.debug("size will change from {0} to {1}", this.expectedSize, expectedSize);
                    this.expectedSize = expectedSize;
                }
                // calcul instance to create (or delete)
                int unUsedPoolItem = availablePoolItemList.size();
                int total = usedPoolItem + unUsedPoolItem;
                instanceToCreate = expectedSize - total;
                if (instanceToCreate < 0) {
                    // we have to many instances (but we cannot delete busy
                    // instances)
                    int instanceToDelete = Math.min(-instanceToCreate, unUsedPoolItem);
                    for (int i = 0; i < instanceToDelete; i++) {
                        availablePoolItemList.remove(accessManager.choosePoolItemToRelease(unmodifiableAvailablePoolItemList));
                    }
                }
            } finally {
                lock.unlock();
            }

            // we have one less instance to create per factory working
            instanceToCreate -= scheduledCreateItemCount;
            if (instanceToCreate <= 0) {
                // try to cancel useless creation
                int cancelled = createItemActionScheduler.cancelAction(-instanceToCreate);
                // remove canceled from scheduled count
                scheduledCreateItemCount -= cancelled;
                return;
            }
            scheduledCreateItemCount += instanceToCreate;
        }
        // ask to call 'instanceToCreate' times the createNewItem() method
        createItemActionScheduler.scheduleAction(instanceToCreate);
    }

    /**
     * @return true if the call to poolItemFactory was done false otherwise
     */
    protected boolean createNewItem() {
        int instanceToCreate = 0;

        synchronized (sizeMutex) {
            lock.lock();
            try {
                // calcul instance to create (or delete)
                int unUsedPoolItem = availablePoolItemList.size();
                int total = usedPoolItem + unUsedPoolItem;
                instanceToCreate = expectedSize - total;
                if (instanceToCreate <= 0) {
                    // we have to many instances (however we should manage remove)
                    if (instanceToCreate != 0 && unUsedPoolItem != 0) {
                        LOG.error("Issue in algorithm : check remove / put methods");
                    }
                    // creation is useless (cancel operation was called too late)
                    scheduledCreateItemCount--;
                    return false;
                }
            } finally {
                lock.unlock();
            }
        }

        RecorderPoolItem<E, I> poolItem = new RecorderPoolItem<E, I>(poolItemFactory.createPoolItem());
        LOG.debug("create new item");

        synchronized (recordMutex) {
            synchronized (sizeMutex) {

                lock.lock();
                try {
                    availablePoolItemList.add(accessManager.createPoolItem(unmodifiableAvailablePoolItemList, poolItem), poolItem);
                    // calcul instance to create (or delete)
                    int unUsedPoolItem = availablePoolItemList.size();
                    int total = usedPoolItem + unUsedPoolItem;
                    instanceToCreate = this.expectedSize - total;
                    if (instanceToCreate < 0) {
                        // we just create an useless instance
                        int instanceToDelete = Math.min(-instanceToCreate, unUsedPoolItem);
                        if (instanceToDelete != 1) {
                            LOG.error("Issue in algorithm : check remove / put methods");
                        }
                        availablePoolItemList.remove(accessManager.choosePoolItemToRelease(unmodifiableAvailablePoolItemList));
                    } else {
                        // our new instance can be use
                        signalClearableCondition.signal();
                    }
                    if (delayedCount != 0) {
                        // at least a thread wait while some item are available
                        // the reason of that is the DELAY_GET, we have to signalAll thread in this case
                        // however thanks to lock implementation (clearAllSignal) all thread will not necessary be wake up.
                        signalClearableCondition.signalAll();
                    }

                } finally {
                    lock.unlock();
                }
                scheduledCreateItemCount--;
            }
        }
        return true;

    }

    /**
     * This method block is no instance are available.
     * @param timeout after this timeout a exception is thrown
     * @return a pool item
     * @throws TimeoutPoolException if timeout
     */
    public E get(final long timeout) throws TimeoutPoolException, IllegalTimeoutException, WaiterInterruptedException {
        if (timeout < 0 && timeout != INFINITE_TIMEOUT) {
            throw new IllegalTimeoutException();
        }
        RecorderPoolItem<E, I> poolItem;
        E item;
        boolean delayed = false;
        lock.lock();
        try {
            int size = availablePoolItemList.size();
            int poolItemToGetChoosed = IAccessManager.DELAY_GET;
            if (size != 0) {
                poolItemToGetChoosed = accessManager.choosePoolItemToGet(unmodifiableAvailablePoolItemList);
                if (poolItemToGetChoosed == IAccessManager.DELAY_GET) {
                    delayedCount++;
                    delayed = true;
                }
            }
            if (poolItemToGetChoosed == IAccessManager.DELAY_GET) {
                long expireTime = 0;
                long timeToWait;
                if (timeout == INFINITE_TIMEOUT) {
                    timeToWait = 0;
                } else {
                    long currentTime = System.currentTimeMillis();
                    expireTime = currentTime + timeout;
                    timeToWait = expireTime - currentTime;
                    if (timeToWait <= 0) {
                        throw new TimeoutPoolException();
                    }
                }
                while (poolItemToGetChoosed == IAccessManager.DELAY_GET) {
                    if (interruptingAllWaiters) {
                        throw new WaiterInterruptedException();
                    }
                    waiterCount++;
                    LOG.debug("nb waiter increase to {0}", waiterCount);
                    try {
                        signalClearableCondition.await(timeToWait);
                        LOG.debug("wait notified or expired");
                    } catch (InterruptedException e) {
                        LOG.debug("wait interrupted");
                    }
                    waiterCount--;
                    LOG.debug("nb waiter decrease to {0}", waiterCount);
                    if (interruptingAllWaiters && waiterCount == 0) {
                        signalClearableCondition.signal();
                    }
                    size = availablePoolItemList.size();
                    if (size != 0) {
                        poolItemToGetChoosed = accessManager.choosePoolItemToGet(unmodifiableAvailablePoolItemList);
                        if (!delayed && poolItemToGetChoosed == IAccessManager.DELAY_GET) {
                            delayedCount++;
                            delayed = true;
                        }
                    } else {
                        if (delayed) {
                            // presume not delayed if no item available
                            delayedCount--;
                            delayed = false;
                        }
                        if (timeout != INFINITE_TIMEOUT) {
                            timeToWait = expireTime - System.currentTimeMillis();
                            if (timeToWait <= 0) {
                                throw new TimeoutPoolException();
                            }
                        }
                    }
                }
            }
            usedPoolItem++;
            poolItem = availablePoolItemList.remove(poolItemToGetChoosed);
            if (size == 1) {
                // no more item to distribute so we avoid wake up of other threads.
                signalClearableCondition.clearAllSignal();
            }
            LOG.debug("pool {0}/{1}", usedPoolItem, availablePoolItemList.size() + usedPoolItem);
        } finally {
            if (delayed) {
                delayedCount--;
            }
            lock.unlock();
        }
        switch (poolItem.getState()) {
        case FETCHED:
        case RECORDING:
            break;
        case RECORDED:
            // TODO : add item to fetch list
            fetchItemActionScheduler.scheduleAction(1);
        case FETCHING:
            // TODO wait state change to NORMAL or throw timeout exception
            // wait on another mutex ?
            // TODO : if timeout remove item from fetch list
            // maybe the fetchr is working on item so the list may not contain the item
            break;
        default:
            throw new Error("Unknow pool item state");
        }
        item = poolItem.getItem();
        synchronized (busyPoolItem) {
            poolItem.setItem(null);
            busyPoolItem.put(item, poolItem);
        }
        return item;
    }

    /**
     * @return a non null instance
     * @throws WaiterInterruptedException if error occurs
     */
    public E get() throws WaiterInterruptedException {
        try {
            return get(INFINITE_TIMEOUT);
        } catch (TimeoutPoolException e) {
            throw new Error("get(INFINITE_TIMEOUT) should not return a TimeoutPoolException");
        } catch (IllegalTimeoutException e) {
            throw new Error("get(INFINITE_TIMEOUT) should not return a IllegalTimeoutException");
        }
    }

    /**
     * This method will never block. If the pool owns too many instances,
     * release method will be apply on pool item.
     * @param poolItem a pool item
     * @throws NotABusyPoolItemException if error occurs
     */
    public void put(final E poolItem) throws NotABusyPoolItemException {
        if (poolItem == null) {
            throw new NotABusyPoolItemException();
        }
        RecorderPoolItem<E, I> recorderPoolItem;
        synchronized (busyPoolItem) {
            recorderPoolItem = busyPoolItem.remove(poolItem);
            if (recorderPoolItem == null) {
                throw new NotABusyPoolItemException();
            }
            recorderPoolItem.setItem(poolItem);
        }
        lock.lock();
        try {
            int remainToDelete = usedPoolItem - expectedSize;
            usedPoolItem--;
            availablePoolItemList.add(accessManager.putPoolItem(unmodifiableAvailablePoolItemList, recorderPoolItem), recorderPoolItem);
            if (remainToDelete <= 0) {
                // add to available if no remain to delete
                if (delayedCount == 0) {
                    signalClearableCondition.signal();
                } else {
                    // at least a thread wait while some item are available
                    // the reason of that is the DELAY_GET, we have to signalAll thread in this case
                    // however thanks to lock implementation (clearAllSignal) all thread will not necessary be wake up.
                    signalClearableCondition.signalAll();
                }
            } else {
                // let gc run on poolItem otherwise
                availablePoolItemList.remove(accessManager.choosePoolItemToRelease(unmodifiableAvailablePoolItemList));
            }
            LOG.debug("pool {0}/{1} (used/total)", usedPoolItem, availablePoolItemList.size() + usedPoolItem);
        } finally {
            lock.unlock();
        }
    }

    /**
     * @return the poolItemFactory
     */
    protected IPoolItemFactory<? extends E> getPoolItemFactory() {
        return poolItemFactory;
    }

    /**
     * @param usedPoolItem the usedPoolItem to set
     */
    protected void setUsedPoolItem(final int usedPoolItem) {
        this.usedPoolItem = usedPoolItem;
    }

    /**
     * @return the mutex to sync before calling protected methods
     */
    protected List<RecorderPoolItem<E, I>> getAvailablePoolItemList() {
        return availablePoolItemList;
    }

    /**
     * @return the expectedSize
     */
    protected int getExpectedSize() {
        return expectedSize;
    }

    /**
     * @return the usedPoolItem
     */
    protected int getUsedPoolItem() {
        return usedPoolItem;
    }

    /**
     * @return the unUsedPoolItem
     */
    protected int getUnUsedPoolItem() {
        return availablePoolItemList.size();
    }

    /**
     * @return the waiterCount
     */
    protected int getWaiterCount() {
        return waiterCount;
    }

    /**
     * @param waiterCount the waiterCount to set
     */
    protected void setWaiterCount(final int waiterCount) {
        this.waiterCount = waiterCount;
    }

    /**
     * @return the interruptingAllWaiters
     */
    protected boolean isInterruptingAllWaiters() {
        return interruptingAllWaiters;
    }

    public void interruptAllWaiters() {
        lock.lock();
        try {
            LOG.debug("begin interrupt {0}/{1} (used/total)", usedPoolItem, availablePoolItemList.size() + usedPoolItem);
            interruptingAllWaiters = true;
            signalClearableCondition.signalAll();
            while (waiterCount != 0) {
                try {
                    signalClearableCondition.await();
                } catch (InterruptedException e) {
                    LOG.debug("wait interrupted", e);
                }
            }
            interruptingAllWaiters = false;
            LOG.debug("end interrupt {0}/{1} (used/total)", usedPoolItem, availablePoolItemList.size() + usedPoolItem);
        } finally {
            lock.unlock();
        }
    }

    public void remove(final E poolItem) throws NotABusyPoolItemException {
        if (poolItem == null) {
            throw new NotABusyPoolItemException();
        }
        boolean scheduleCreation = false;
        RecorderPoolItem<E, I> recorderPoolItem;
        synchronized (busyPoolItem) {
            recorderPoolItem = busyPoolItem.remove(poolItem);
            if (recorderPoolItem == null) {
                throw new NotABusyPoolItemException();
            }
        }

        synchronized (sizeMutex) {
            int instanceToCreate = 0;

            lock.lock();
            try {
                if (usedPoolItem <= 0) {
                    throw new NoBusyPoolItemException();
                }
                accessManager.removePoolItem(recorderPoolItem);
                usedPoolItem--;
                // calcul instance to create (or delete)
                int unUsedPoolItem = availablePoolItemList.size();
                int total = usedPoolItem + unUsedPoolItem;
                instanceToCreate = expectedSize - total;
                if (instanceToCreate < 0 && unUsedPoolItem != 0) {
                    LOG.error("issue in setExpectedSize, remove method should not make this stuff");
                }
                if (LOG.isDebugEnabled()) {
                    LOG.debug("pool {0}/{1} (used/total)", usedPoolItem, availablePoolItemList.size() + usedPoolItem);
                }
            } finally {
                lock.unlock();
            }

            // we have one less instance to create per factory working
            instanceToCreate -= scheduledCreateItemCount;
            if (instanceToCreate > 0) {
                scheduledCreateItemCount++;
                scheduleCreation = true;
                if (instanceToCreate != 1) {
                    LOG.error("issue in setExpectedSize, remove method should not schedule more than one instance creation");
                }
            }
        }
        if (scheduleCreation) {
            createItemActionScheduler.scheduleAction(1);
        }
    }


}
