/**
 * OW2 Util
 * Copyright (C) 2009 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 * --------------------------------------------------------------------------
 * $Id: LastUsedRecorderAccessManager.java 4809 2009-03-15 10:13:49Z gaellalire $
 * --------------------------------------------------------------------------
 */

package org.ow2.util.pool.impl.enhanced.impl.recorder.accesmanager;

import java.util.List;

import org.ow2.util.pool.impl.enhanced.api.recorder.IRecorderPoolItem;
import org.ow2.util.pool.impl.enhanced.api.recorder.RecorderPoolItemState;
import org.ow2.util.pool.impl.enhanced.api.recorder.accesmanager.IRecorderAccessManager;

/**
 * @author Gael Lalire
 */
public class LastUsedRecorderAccessManager<E, I> implements IRecorderAccessManager<E, I> {

    private int fetchedSize;

    private int fetchingSize;

    private int recordingSize;

    private int recordedSize;

    public int choosePoolItemToGet(final List<? extends IRecorderPoolItem<? extends E, ? extends I>> poolItemList) {
        fetchedSize--;
        return firstFetchedPosition();
    }

    public int putPoolItem(final List<? extends IRecorderPoolItem<? extends E, ? extends I>> poolItemList, final IRecorderPoolItem<? extends E, ? extends I> poolItem) {
        fetchedSize++;
        return firstFetchedPosition();
    }

    public int choosePoolItemToRelease(final List<? extends IRecorderPoolItem<? extends E, ? extends I>> poolItemList) {
        return lastRecordedPosition();
    }

    public int createPoolItem(final List<? extends IRecorderPoolItem<? extends E, ? extends I>> poolItemList, final IRecorderPoolItem<? extends E, ? extends I> poolItem) {
        fetchedSize++;
        return lastFetchedPosition();
    }

    public int choosePoolItemToFetch(final List<? extends IRecorderPoolItem<? extends E, ? extends I>> poolItemList) {
        return firstRecordedPosition();
    }

    public int choosePoolItemToRecord(final List<? extends IRecorderPoolItem<? extends E, ? extends I>> poolItemList) {
        return lastFetchedPosition();
    }

    public int movePoolItemWhoseStateChanged(final List<? extends IRecorderPoolItem<? extends E, ? extends I>> poolItemList,
            final IRecorderPoolItem<? extends E, ? extends I> poolItem, final RecorderPoolItemState previousState) {
        int previousPosition = poolItemList.indexOf(poolItem);
        switch (previousState) {
        case FETCHED:
            fetchedSize--;
            break;
        case FETCHING:
            fetchingSize--;
            break;
        case RECORDING:
            recordingSize--;
            break;
        case RECORDED:
            recordedSize--;
            break;
        default:
            throw new Error();
        }
        switch (poolItem.getState()) {
        case FETCHED:
            // come from either FETCHING or an aborted RECORDING (after => last)
            fetchedSize++;
            return lastFetchedPosition() - previousPosition;
        case FETCHING:
            // come from RECORDED (after => last)
            fetchingSize++;
            return lastFetchingPosition() - previousPosition;
        case RECORDING:
            // come from FETCHED (before => first)
            recordingSize++;
            return firstRecordingPosition() - previousPosition;
        case RECORDED:
            // come from RECORDING (before => first)
            recordedSize++;
            return firstRecordedPosition() - previousPosition;
        default:
            throw new Error();
        }
    }

    public void removePoolItem(final IRecorderPoolItem<? extends E, ? extends I> poolItem) {
    }

    private int firstFetchedPosition() {
        return 0;
    }

    private int lastFetchedPosition() {
        return fetchedSize - 1;
    }

    private int firstFetchingPosition() {
        return fetchedSize;
    }

    private int lastFetchingPosition() {
        return fetchedSize + fetchingSize - 1;
    }

    private int firstRecordingPosition() {
        return fetchedSize + fetchingSize;
    }

    private int lastRecordingPosition() {
        return fetchedSize + fetchingSize + recordingSize - 1;
    }

    private int firstRecordedPosition() {
        return fetchedSize + fetchingSize + recordingSize;
    }

    private int lastRecordedPosition() {
        return fetchedSize + fetchingSize + recordingSize + recordedSize - 1;
    }

}
