/**
 * OW2 Util
 * Copyright (C) 2009 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 * --------------------------------------------------------------------------
 * $Id: ActionScheduler.java 4601 2009-02-16 09:35:38Z gaellalire $
 * --------------------------------------------------------------------------
 */

package org.ow2.util.pool.impl.enhanced.internal.actionscheduler;

import java.util.Iterator;
import java.util.LinkedList;
import java.util.concurrent.Callable;
import java.util.concurrent.Executor;
import java.util.concurrent.FutureTask;


/**
 *
 * @author Gael Lalire
 */
public class ActionScheduler implements IActionScheduler {

    private static final int ELEMENT_TO_KEEP_FOR_AVERAGE_COMPUTATION = 20;

    private LinkedList<FutureTask<Boolean>> scheduledCallableList;

    private Callable<Boolean> action;

    private Executor executor;

    private Object mutex = new Object();

    private Average average = new Average(ELEMENT_TO_KEEP_FOR_AVERAGE_COMPUTATION);

    public ActionScheduler(final Callable<Boolean> action, final Executor executor) {
        this.executor = executor;
        this.action = action;
        scheduledCallableList = new LinkedList<FutureTask<Boolean>>();
    }

    public void scheduleAction(final int time) {
        for (int i = 0; i < time; i++) {
            FutureTask<Boolean> futureTask = new FutureTask<Boolean>(action) {

                private long startTime;

                @Override
                public void run() {
                    startTime = System.currentTimeMillis();
                    super.run();
                }

                @Override
                protected void done() {

                    if (!isCancelled()) {
                        average.addElement(System.currentTimeMillis() - startTime);
                        synchronized (mutex) {
                            scheduledCallableList.remove(this);
                        }
                    }

                }

            };
            synchronized (mutex) {
                scheduledCallableList.addFirst(futureTask);
            }
            executor.execute(futureTask);
        }
    }

    public int cancelAction(final int time) {
        synchronized (mutex) {
            int canceledSheduledCallable = 0;
            Iterator<FutureTask<Boolean>> iterator = scheduledCallableList.iterator();
            while (iterator.hasNext() && canceledSheduledCallable < time) {
                FutureTask<Boolean> futureTask = iterator.next();
                if (futureTask.cancel(false)) {
                    canceledSheduledCallable++;
                    iterator.remove();
                    // do not remove from list cause done() is called even if canceled.
                }
            }
            return canceledSheduledCallable;
        }
    }

}
