/**
 * OW2 Util
 * Copyright (C) 2008 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 * --------------------------------------------------------------------------
 * $Id: BasicCondition.java 4759 2009-03-08 19:05:13Z gaellalire $
 * --------------------------------------------------------------------------
 */

package org.ow2.util.pool.impl.enhanced.internal.lock.impl;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.LockSupport;

import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;
import org.ow2.util.pool.impl.enhanced.internal.lock.api.ISignalClearableCondition;

/**
 *
 * @author Gael Lalire
 */
public class BasicCondition implements ISignalClearableCondition {

    private static final Log LOG = LogFactory.getLog(BasicCondition.class);

    private final ILockWithSignalClearableConditionControl lockControl;

    private List<Thread> unsignaledThread = new ArrayList<Thread>();

    private List<Thread> signaledThread = new ArrayList<Thread>();

    /**
     * Only one waiting thread which is unpark.
     */
    private volatile Thread effectivelySignaledThread;

    public BasicCondition(final ILockWithSignalClearableConditionControl lockControl) {
        this.lockControl = lockControl;
    }

    /**
     * @param time in milliseconds if 0 then no timeout
     * @throws InterruptedException
     */
    public void await(final long time) throws InterruptedException {
        lockControl.checkOwner();
        long until = System.currentTimeMillis() + time;
        Thread current = Thread.currentThread();

        unsignaledThread.add(current);

        lockControl.innerUnlock();

        // !!! Not owner of lock

        try {
            boolean ended = false;
            do {
                if (Thread.interrupted()) {
                    throw new InterruptedException();
                }
                if (time == 0) {
                    LockSupport.park();
                    if (effectivelySignaledThread == current) {
                        ended = true;
                    }
                } else {
                    LockSupport.parkUntil(until);
                    if (effectivelySignaledThread == current || System.currentTimeMillis() > until) {
                        ended = true;
                    }
                }
            } while (!ended);
        } finally {
            // even on InterruptedException we have to get back the lock

            lockControl.innerLock();

            if (effectivelySignaledThread == current) {
                effectivelySignaledThread = null;
            } else {
                if (!signaledThread.remove(current)) {
                    if (!unsignaledThread.remove(current)) {
                        throw new Error("Lock issue");
                    }
                }
            }

        }

        // Owner of lock
    }

    public void signalAll() {
        lockControl.checkOwner();
        if (!unsignaledThread.isEmpty()) {
            signaledThread.addAll(unsignaledThread);
            unsignaledThread.clear();
            lockControl.askForUnparkOneThread(signalClearableConditionControl);
        }
    }


    public void signal() {
        lockControl.checkOwner();
        if (!unsignaledThread.isEmpty()) {
            signaledThread.add(unsignaledThread.remove(0));
            lockControl.askForUnparkOneThread(signalClearableConditionControl);
        }
    }

    public void clearAllSignal() {
        lockControl.checkOwner();
        LOG.debug("clearAllSignal");

        if (!signaledThread.isEmpty()) {
            unsignaledThread.addAll(signaledThread);
            signaledThread.clear();
            lockControl.avoidAskForUnparkOneThread(signalClearableConditionControl);
        }
    }

    public void await() throws InterruptedException {
        await(0);
    }

    public boolean await(final long time, final TimeUnit unit) throws InterruptedException {
        long millisTimeout = unit.toMillis(time);
        if (millisTimeout <= 0) {
            return false;
        }
        await(millisTimeout);
        return true;
    }

    public long awaitNanos(final long nanosTimeout) throws InterruptedException {
        throw new UnsupportedOperationException();
    }

    public void awaitUninterruptibly() {
        throw new UnsupportedOperationException();
    }

    public boolean awaitUntil(final Date deadline) throws InterruptedException {
        throw new UnsupportedOperationException();
    }

    AtomicInteger atomicInteger = new AtomicInteger(2000);

    private ISignalClearableConditionControl signalClearableConditionControl = new ISignalClearableConditionControl() {

        public boolean unparkOneThread() {
            if (effectivelySignaledThread == null) {
                effectivelySignaledThread = signaledThread.remove(0);
                LockSupport.unpark(effectivelySignaledThread);
            }
            if (signaledThread.size() == 0) {
                return false;
            }
            return true;
        }

    };

}
