/**
 * OW2 Util
 * Copyright (C) 2008 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 * --------------------------------------------------------------------------
 * $Id: BasicLock.java 4759 2009-03-08 19:05:13Z gaellalire $
 * --------------------------------------------------------------------------
 */

package org.ow2.util.pool.impl.enhanced.internal.lock.impl;

import java.util.HashSet;
import java.util.Iterator;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.LockSupport;

import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * This lock is a non reentrent lock which delay signal and signalAll to optimize
 * processor use. In addition to that it provides a clearAllSignal method.
 * @author Gael Lalire
 */
public class BasicLock implements Lock {

    private static final Log LOG = LogFactory.getLog(BasicLock.class);

    private AtomicBoolean locked = new AtomicBoolean(false);

    private Queue<Thread> waiters = new ConcurrentLinkedQueue<Thread>();

    private volatile Thread owner = null;

    private ILockWithSignalClearableConditionControl lockControl = new ILockWithSignalClearableConditionControl() {

        private Set<ISignalClearableConditionControl> toUnparkConditionSet = new HashSet<ISignalClearableConditionControl>();

        public void innerLock() {
            LOG.debug("ask for lock {0}", BasicLock.this);
            boolean wasInterrupted = false;

            Thread current = Thread.currentThread();
            if (current == owner) {
                throw new IllegalStateException("This lock is not reentrent lock");
            }

            waiters.add(current);

            while (waiters.peek() != current || !locked.compareAndSet(false, true)) {
                LockSupport.park();
                if (Thread.interrupted()) {
                    wasInterrupted = true;
                }
            }
            LOG.debug("lock obtained {0}", BasicLock.this);

            waiters.remove(current);
            if (wasInterrupted) {
                current.interrupt();
            }
            owner = current;
        }

        public void innerUnlock() {
            Iterator<ISignalClearableConditionControl> it = toUnparkConditionSet.iterator();
            if (it.hasNext()) {
                if (!it.next().unparkOneThread()) {
                    it.remove();
                }
            }
            owner = null;
            LOG.debug("lock {0} release", BasicLock.this);

            locked.set(false);
            LockSupport.unpark(waiters.peek());
        }

        public void checkOwner() {
            if (Thread.currentThread() != owner) {
                throw new IllegalMonitorStateException();
            }
        }

        public void askForUnparkOneThread(final ISignalClearableConditionControl basicCondition) {
            toUnparkConditionSet.add(basicCondition);
        }

        public void avoidAskForUnparkOneThread(final ISignalClearableConditionControl basicCondition) {
            toUnparkConditionSet.remove(basicCondition);
        }

    };

    public void lock() {
        lockControl.innerLock();
    }

    public void unlock() {
        lockControl.checkOwner();
        lockControl.innerUnlock();
    }

    public void lockInterruptibly() throws InterruptedException {
        if (Thread.interrupted()) {
            throw new InterruptedException();
        }
        lockControl.innerLock();
    }

    public BasicCondition newCondition() {
        return new BasicCondition(lockControl);
    }

    public boolean tryLock() {
        throw new UnsupportedOperationException();
    }

    public boolean tryLock(final long time, final TimeUnit unit) throws InterruptedException {
        throw new UnsupportedOperationException();
    }

}
