/**
 * OW2 Util
 * Copyright (C) 2008 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 * --------------------------------------------------------------------------
 * $Id: MinMaxExpectedSpareStockerResizer.java 4389 2008-12-15 13:48:57Z alitokmen $
 * --------------------------------------------------------------------------
 */

package org.ow2.util.pool.impl.enhanced.internal.resizer.impl.stocker;

import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;
import org.ow2.util.pool.impl.enhanced.internal.resizer.api.statistics.StockerStatistics;
import org.ow2.util.pool.impl.enhanced.internal.resizer.impl.AbstractResizer;


/**
 *
 * @author Gael Lalire
 */
public class MinMaxExpectedSpareStockerResizer<S extends StockerStatistics> extends AbstractResizer<S> {

    private static final Log LOG = LogFactory.getLog(MinMaxExpectedSpareStockerResizer.class);

    public static final int NO_MAX_POOL_LIMIT = -1;

    private boolean noRegression;

    private int expectedSize;

    private int minPool;

    private int maxPool;

    private int expectedSparePool;

    private Object mutex;

    public MinMaxExpectedSpareStockerResizer(final int minPool, final int maxPool, final boolean noRegression) {
        this(minPool, maxPool, 1, noRegression);
    }

    public MinMaxExpectedSpareStockerResizer(final int minPool, final int maxPool, final int expectedSparePool,
            final boolean noRegression) {
        if (minPool < 0 || expectedSparePool < 1) {
            throw new IllegalArgumentException();
        }
        if (maxPool != NO_MAX_POOL_LIMIT) {
            if (minPool > maxPool || maxPool < 0) {
                throw new IllegalArgumentException();
            }
        }
        this.minPool = minPool;
        this.maxPool = maxPool;
        this.expectedSparePool = expectedSparePool;
        this.noRegression = noRegression;
        mutex = new Object();
    }

    /**
     * This function is called just after a pool item is get or released. So if
     * you set expectedSize with same or less value than usedPoolItem the pool
     * will be blocking.
     */
    public void prepareUpdate(final S stockedCount) {
        int usedPoolItem = stockedCount.getStockedCount();
        synchronized (mutex) {
            LOG.debug("prepare an update {0}, conf min {1} max {2} spare {3}", usedPoolItem, minPool, maxPool,
                    expectedSparePool);
            int expectedTotal = usedPoolItem + expectedSparePool;
            if (noRegression && expectedTotal < this.expectedSize) {
                expectedTotal = this.expectedSize;
            }
            if (expectedTotal < minPool) {
                expectedSize = minPool;
            } else if (maxPool != NO_MAX_POOL_LIMIT && expectedTotal > maxPool) {
                expectedSize = maxPool;
            } else {
                expectedSize = expectedTotal;
            }
        }
    }

    public void update() {
        int expectedSize;
        synchronized (mutex) {
            expectedSize = this.expectedSize;
        }
        LOG.debug("call change size to {0} by {1}", expectedSize, Thread.currentThread());
        updateWith(expectedSize);
    }

    /**
     * @param expectedSparePool the expectedSparePool to set
     */
    public void setExpectedSparePool(final int expectedSparePool) {
        if (maxPool != NO_MAX_POOL_LIMIT) {
            if (expectedSparePool < 1) {
                throw new IllegalArgumentException();
            }
        }
        synchronized (mutex) {
            if (this.expectedSparePool != expectedSparePool) {
                this.expectedSparePool = expectedSparePool;
            }
        }
    }

    public void setMaxPool(final int maxPool) {
        if (maxPool != NO_MAX_POOL_LIMIT) {
            if (maxPool < 0) {
                throw new IllegalArgumentException();
            }
            synchronized (mutex) {
                if (minPool > maxPool) {
                    minPool = maxPool;
                }
                this.maxPool = maxPool;
            }
        } else {
            synchronized (mutex) {
                this.maxPool = maxPool;
            }
        }
    }

    public void setMinPool(final int minPool) {
        if (minPool < 0) {
            throw new IllegalArgumentException();
        }
        synchronized (mutex) {
            if (minPool > maxPool) {
                maxPool = minPool;
            }
            this.minPool = minPool;
        }
    }

}
