/**
 * OW2 Util
 * Copyright (C) 2008 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 * --------------------------------------------------------------------------
 * $Id: CleanerReferencer.java 4389 2008-12-15 13:48:57Z alitokmen $
 * --------------------------------------------------------------------------
 */

package org.ow2.util.pool.impl.enhanced.internal.thread.impl;

import java.lang.ref.Reference;
import java.lang.ref.ReferenceQueue;
import java.lang.ref.WeakReference;

/**
 * Keep a reference on cleaner.
 * If cleaner does not have anything to clean then we keep only
 * a weak reference on cleaner. So if cleaner is not used anywhere else,
 * the cleaner is garbage collected. It produce the end of the thread running
 * this object. And normally this object should be also garbage collected.
 *
 * Warning : if you do not call add after creating a weak reference which use
 * queue of cleaner, then the thread running keeper could be terminated whereas
 * cleaner is not garbage collected. The cleaner will not continue to work but no
 * error will occurs.
 * @author Gael Lalire
 */
public class CleanerReferencer extends WeakReference<Cleaner> implements Runnable {

    private int instanceToClean;

    @SuppressWarnings("unused")
    private Cleaner cleaner;

    private Thread thread;

    private ReferenceQueue<Object> referenceQueue;

    private Object mutex;

    public CleanerReferencer(final Cleaner cleaner) {
        super(cleaner, cleaner.getReferenceQueue());
        mutex = new Object();
        this.referenceQueue = cleaner.getReferenceQueue();
        thread = new Thread(this);
        thread.setDaemon(true);
        thread.start();
        instanceToClean = 0;
    }

    public void run() {
        while (true) {
            try {
                Reference<?> reference = referenceQueue.remove();
                if (reference instanceof ICleanable) {
                    ICleanable cleanable = (ICleanable) reference;
                    try {
                        cleanable.clean();
                    } catch (RuntimeException e) {
                        e.printStackTrace();
                    }
                }
                if (remove()) {
                    break;
                }
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        }
    }

    public void add() {
        synchronized (mutex) {
            instanceToClean++;
            cleaner = get();
        }
    }

    private boolean remove() {
        synchronized (mutex) {
            if (instanceToClean < 0) {
                throw new IllegalStateException();
            }
            if (instanceToClean == 0) {
                return true;
            }
            instanceToClean--;
            if (instanceToClean == 0) {
                cleaner = null;
            }
            return false;
        }
    }

}
