/**
 * OW2 Util
 * Copyright (C) 2008 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 * --------------------------------------------------------------------------
 * $Id: ManagementThread.java 4389 2008-12-15 13:48:57Z alitokmen $
 * --------------------------------------------------------------------------
 */

package org.ow2.util.pool.impl.enhanced.internal.thread.impl;

import org.ow2.util.pool.impl.enhanced.internal.thread.api.IManagementThread;

/**
 * Runnable can be restarted. In addition the runnable object can be garbage
 * collected. When the runnable is garbage collected this thread terminates.
 * @author Gael Lalire
 */
public class ManagementThread extends CleanableWeakReference<Runnable> implements Runnable, IManagementThread {

    /**
     * When the runnable will be garbage collected, the
     * cleaner will call clean method.
     */
    private static Cleaner cleaner;

    static {
        cleaner = new Cleaner();
    }

    private boolean updateNeeded;

    private Object mutex;

    public ManagementThread(final Runnable runnable) {
        this(runnable, true);
    }

    public ManagementThread(final Runnable runnable, final boolean daemon) {
        super(runnable, cleaner);
        updateNeeded = false;
        mutex = new Object();
        Thread thread = new Thread(this);
        thread.setDaemon(daemon);
        thread.start();
    }


    public void run() {
        mainloop: while (true) {
            Runnable runnable = get();
            if (runnable == null) {
                break mainloop;
            }
            synchronized (mutex) {
                while (!updateNeeded) {
                    runnable = null;
                    try {
                        mutex.wait();
                    } catch (InterruptedException e) {
                    }
                    runnable = get();
                    if (runnable == null) {
                        break mainloop;
                    }
                }
                updateNeeded = false;
            }
            // if start here then update will be called after this time
            // but the only way to avoid it is to keep mutex.
            // This is a bad solution because start method will be
            // slower.
            runnable.run();
        }
    }

    public void start() {
        synchronized (mutex) {
            updateNeeded = true;
            mutex.notify();
        }
    }

    public void clean() {
        synchronized (mutex) {
            mutex.notify();
        }
    }

}
