/*
 * $Id: SocketExceptionHelper.java,v 1.7 2010/09/06 09:19:11 oboehm Exp $
 *
 * Copyright (c) 2010 by Oliver Boehm
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express orimplied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * (c)reated 12.05.2010 by oliver (ob@oasd.de)
 */

package patterntesting.exception.net;

import java.net.*;

import org.apache.commons.lang.StringUtils;

/**
 * The Class SocketExceptionHelper with some utility methods for better
 * SocketExceptions or NoRouteToHostExceptions.
 *
 * @author oliver
 * @since 1.0 (12.05.2010)
 */
public final class SocketExceptionHelper {

    /** No need for to instantiate this class (utility class). */
    private SocketExceptionHelper() {
    }

    /**
     * If the given NoRouteToHostException has only the default message
     * ("No route to host") this message will be replaced by a better one.
     *
     * @param e the original NoRouteToHostException
     * @param connection the connection
     * @return a NoRouteToHostException which contains the host name or address
     */
    public static NoRouteToHostException getBetterNoRouteToHostException(
            final NoRouteToHostException e, final URLConnection connection) {
        String msg = e.getMessage();
        URL url = connection.getURL();
        if ("No route to host".equalsIgnoreCase(msg)) {
            msg = "No route to " + url.getHost();
        } else {
            msg = msg + " (" + url.getHost() + ")";
        }
        return new NoRouteToHostException(msg);
    }

    /**
     * The given SocketException is checked if it contains the host in the
     * error message. If not it is appended.
     *
     * @param e the original SocketException
     * @param connection the URLConnection
     * @return a better SocketException if in the original the host is missing
     */
    public static SocketException getBetterSocketException(final SocketException e,
            final URLConnection connection) {
        String msg = e.getMessage();
        URL url = connection.getURL();
        String host = url.getHost();
        if (StringUtils.contains(msg, host)) {
            return e;
        }
        String betterMsg = msg + " (can't connect " + host + ")";
        return new SocketException(betterMsg);
    }

}

