/*
 * Decompiled with CFR 0.152.
 */
package prompto.store.mongo;

import com.mongodb.MongoCommandException;
import com.mongodb.client.ChangeStreamIterable;
import com.mongodb.client.ClientSession;
import com.mongodb.client.MongoChangeStreamCursor;
import com.mongodb.client.MongoCollection;
import com.mongodb.client.model.Aggregates;
import com.mongodb.client.model.Filters;
import com.mongodb.client.model.IndexOptions;
import com.mongodb.client.model.Indexes;
import com.mongodb.client.model.Projections;
import com.mongodb.client.model.Sorts;
import com.mongodb.client.model.UpdateOptions;
import com.mongodb.client.model.changestream.ChangeStreamDocument;
import com.mongodb.client.model.changestream.FullDocument;
import com.mongodb.client.model.changestream.UpdateDescription;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Stack;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;
import org.bson.BsonBinary;
import org.bson.BsonDocument;
import org.bson.BsonInt64;
import org.bson.BsonTimestamp;
import org.bson.BsonType;
import org.bson.Document;
import org.bson.UuidRepresentation;
import org.bson.conversions.Bson;
import org.bson.internal.UuidHelper;
import org.bson.types.Binary;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import prompto.intrinsic.PromptoBinary;
import prompto.intrinsic.PromptoDateTime;
import prompto.intrinsic.PromptoDbId;
import prompto.intrinsic.PromptoDocument;
import prompto.intrinsic.PromptoList;
import prompto.store.IAuditMetadata;
import prompto.store.IAuditRecord;
import prompto.store.IStored;
import prompto.store.mongo.MongoDbIdConverter;
import prompto.store.mongo.MongoStore;
import prompto.store.mongo.StoredDocument;
import prompto.utils.Logger;

public class MongoAuditor {
    static final Logger logger = new Logger();
    static final String AUDIT_RECORDS_PRIMARY_KEY_INDEX_NAME = "primary_key";
    static final String AUDIT_RECORDS_COLLECTION = "auditRecords";
    static final String AUDIT_METADATAS_COLLECTION = "auditMetadatas";
    static final String AUDIT_CONFIGS_COLLECTION = "auditConfigs";
    static final String METADATA_DBID_FIELD_NAME = "metadataDbId";
    static final String INSTANCE_DBID_FIELD_NAME = "instanceDbId";
    static final String UTC_TIMESTAMP_FIELD_NAME = "utcTimestamp";
    private final AtomicBoolean isTerminated = new AtomicBoolean(false);
    final MongoStore store;
    Thread thread;
    AuditMetadata metadata;

    public MongoAuditor(MongoStore store) {
        this.store = store;
    }

    public void start() {
        this.createUniqueIndexIfRequired();
        this.createThread();
        this.startThread();
    }

    private void createUniqueIndexIfRequired() {
        if (!MongoStore.indexExists((MongoCollection<Document>)this.store.db.getCollection(AUDIT_RECORDS_COLLECTION), AUDIT_RECORDS_PRIMARY_KEY_INDEX_NAME)) {
            this.clearDuplicateAuditRecords();
            IndexOptions options = new IndexOptions().unique(true).name(AUDIT_RECORDS_PRIMARY_KEY_INDEX_NAME);
            Bson keys = Indexes.descending((String[])new String[]{METADATA_DBID_FIELD_NAME, INSTANCE_DBID_FIELD_NAME});
            this.store.db.getCollection(AUDIT_RECORDS_COLLECTION).createIndex(keys, options);
        }
    }

    private void clearDuplicateAuditRecords() {
        HashMap uniqueIds = new HashMap();
        MongoCollection collection = this.store.db.getCollection(AUDIT_RECORDS_COLLECTION);
        collection.find().forEach(doc -> {
            UUID uuid;
            Set instanceIds = uniqueIds.computeIfAbsent((UUID)doc.get((Object)METADATA_DBID_FIELD_NAME, UUID.class), key -> new HashSet());
            if (!instanceIds.add(uuid = (UUID)doc.get((Object)INSTANCE_DBID_FIELD_NAME, UUID.class))) {
                collection.deleteOne(Filters.eq((String)"_id", (Object)doc.get((Object)"_id")));
            }
        });
    }

    private void createThread() {
        if (this.thread != null) {
            throw new IllegalStateException("Auditor thread already exists!");
        }
        this.thread = new Thread(this::watchInstancesChanges, "Mongo auditor");
    }

    private void startThread() {
        if (this.thread == null) {
            throw new IllegalStateException("Auditor thread does not exist!");
        }
        if (this.thread.getState() != Thread.State.NEW) {
            throw new IllegalStateException("Auditor thread is already started!");
        }
        this.thread.start();
    }

    private void watchInstancesChanges() {
        this.watchInstancesChanges(true);
    }

    private void watchInstancesChanges(boolean resuming) {
        BsonTimestamp resumeTimestamp;
        List<Bson> filters = Collections.singletonList(Aggregates.match((Bson)Filters.in((String)"operationType", Arrays.asList("insert", "update", "replace", "delete"))));
        ChangeStreamIterable stream = this.store.getInstancesCollection().watch(filters).fullDocument(FullDocument.UPDATE_LOOKUP);
        BsonTimestamp bsonTimestamp = resumeTimestamp = resuming ? this.fetchLastAuditTimestamp() : null;
        if (resumeTimestamp != null) {
            logger.info(() -> "Resuming audit from " + LocalDateTime.ofEpochSecond(resumeTimestamp.getTime(), 0, ZoneOffset.UTC));
            stream = stream.startAtOperationTime(resumeTimestamp);
        } else {
            logger.warn(() -> "Starting audit without a resume timestamp");
        }
        try (MongoChangeStreamCursor cursor = stream.cursor();){
            while (!this.isTerminated.get()) {
                this.consumeChanges((MongoChangeStreamCursor<ChangeStreamDocument<Document>>)cursor);
                try {
                    TimeUnit.MILLISECONDS.sleep(100L);
                }
                catch (InterruptedException interruptedException) {}
            }
        }
        catch (MongoCommandException e) {
            logger.error(() -> "While starting auditor", (Throwable)e);
            if (resumeTimestamp != null && this.isResumingException(e)) {
                this.watchInstancesChanges(false);
            }
            throw e;
        }
    }

    private boolean isResumingException(MongoCommandException e) {
        return e.getErrorCode() == 236 && "ChangeStreamHistoryLost".equals(e.getErrorCodeName()) || e.getErrorCode() == 286 && "NonResumableChangeStreamError".equals(e.getErrorCodeName());
    }

    private void consumeChanges(MongoChangeStreamCursor<ChangeStreamDocument<Document>> cursor) {
        ChangeStreamDocument change;
        while ((change = (ChangeStreamDocument)cursor.tryNext()) != null) {
            this.auditInstanceChange((ChangeStreamDocument<Document>)change);
        }
        return;
    }

    private BsonTimestamp fetchLastAuditTimestamp() {
        Bson filter = Filters.eq((String)"name", (Object)"LAST_AUDIT_TIMESTAMP");
        Document record = (Document)this.store.db.getCollection(AUDIT_CONFIGS_COLLECTION).find(filter).first();
        if (record != null) {
            return (BsonTimestamp)record.get((Object)"timeStamp", BsonTimestamp.class);
        }
        return null;
    }

    private void storeLastAuditTimestamp(ClientSession session, ChangeStreamDocument<Document> change) {
        Document record = new Document();
        record.put("name", (Object)"LAST_AUDIT_TIMESTAMP");
        record.put("timeStamp", (Object)change.getClusterTime());
        Document update = new Document("$set", (Object)record);
        Bson filter = Filters.eq((String)"name", (Object)"LAST_AUDIT_TIMESTAMP");
        this.store.db.getCollection(AUDIT_CONFIGS_COLLECTION).updateOne(filter, (Bson)update, new UpdateOptions().upsert(true));
    }

    private void auditInstanceChange(ChangeStreamDocument<Document> change) {
        this.loadMetadataRecordIfRequired(change);
        this.storeAuditRecord(change);
    }

    private void storeAuditRecord(ChangeStreamDocument<Document> change) {
        try (ClientSession session = this.store.client.startSession();){
            logger.debug(() -> "Auditing change for record " + this.store.convertToDbId(change.getDocumentKey().get((Object)"_id")));
            session.startTransaction();
            switch (change.getOperationType()) {
                case INSERT: {
                    this.storeInsertRecord(session, change);
                    break;
                }
                case UPDATE: 
                case REPLACE: {
                    this.storeUpdateRecord(session, change);
                    break;
                }
                case DELETE: {
                    this.storeDeleteRecord(session, change);
                    break;
                }
                default: {
                    logger.warn(() -> "Unsupported operation: " + change.getOperationType().name());
                }
            }
            this.storeLastAuditTimestamp(session, change);
            session.commitTransaction();
        }
    }

    private void storeInsertRecord(ClientSession session, ChangeStreamDocument<Document> change) {
        AuditRecord insert = this.newAuditRecord();
        insert.setInstanceDbId(this.store.convertToDbId(change.getDocumentKey().get((Object)"_id")));
        insert.setOperation(IAuditRecord.Operation.INSERT);
        insert.setInstance(new StoredDocument(this.store, (Document)change.getFullDocument()));
        this.storeAuditRecord(session, insert);
    }

    private void storeUpdateRecord(ClientSession session, ChangeStreamDocument<Document> change) {
        AuditRecord update = this.newAuditRecord();
        update.setInstanceDbId(this.store.convertToDbId(change.getDocumentKey().get((Object)"_id")));
        update.setOperation(IAuditRecord.Operation.UPDATE);
        update.setInstance(new StoredDocument(this.store, (Document)change.getFullDocument()));
        UpdateDescription description = change.getUpdateDescription();
        update.put("removedFields", description == null ? null : description.getRemovedFields());
        update.put("updatedFields", description == null ? null : description.getUpdatedFields());
        this.storeAuditRecord(session, update);
    }

    private void storeDeleteRecord(ClientSession session, ChangeStreamDocument<Document> change) {
        AuditRecord delete = this.newAuditRecord();
        delete.setInstanceDbId(this.store.convertToDbId(change.getDocumentKey().get((Object)"_id")));
        delete.setOperation(IAuditRecord.Operation.DELETE);
        this.storeAuditRecord(session, delete);
    }

    private void storeAuditRecord(ClientSession session, AuditRecord record) {
        Object dbId = record.remove("_id");
        Document update = new Document("$set", (Object)record);
        update.put("$setOnInsert", (Object)new Document("_id", dbId));
        PromptoDbId metaId = record.getMetadataDbId();
        if (metaId == null) {
            metaId = this.store.newDbId();
        }
        Bson filter = Filters.and((Bson[])new Bson[]{Filters.eq((String)METADATA_DBID_FIELD_NAME, (Object)metaId.getValue()), Filters.eq((String)INSTANCE_DBID_FIELD_NAME, (Object)record.getInstanceDbId().getValue())});
        this.store.db.getCollection(AUDIT_RECORDS_COLLECTION).updateOne(filter, (Bson)update, new UpdateOptions().upsert(true).hintString(AUDIT_RECORDS_PRIMARY_KEY_INDEX_NAME));
        record.put("_id", dbId);
    }

    private AuditRecord newAuditRecord() {
        AuditRecord audit = new AuditRecord(this.store);
        audit.setDbId(this.store.convertToDbId(UUID.randomUUID()));
        if (this.metadata != null) {
            audit.setMetadataDbId(this.metadata.getAuditMetadataId(this.store));
            audit.setUTCTimestamp(this.metadata.getUTCTimestamp());
        } else {
            audit.setMetadataDbId(null);
            audit.setUTCTimestamp(LocalDateTime.now());
        }
        return audit;
    }

    private void loadMetadataRecordIfRequired(ChangeStreamDocument<Document> change) {
        BsonDocument lsId = change.getLsid();
        if (lsId == null) {
            this.metadata = this.createMetadata();
        } else {
            Object sessionId = null;
            if (lsId.get((Object)"id").getBsonType() == BsonType.BINARY) {
                BsonBinary binary = lsId.get((Object)"id").asBinary();
                sessionId = UuidHelper.decodeBinaryToUuid((byte[])binary.getData(), (byte)binary.getType(), (UuidRepresentation)UuidRepresentation.STANDARD);
            } else {
                sessionId = lsId.toString();
            }
            BsonInt64 txnNumber = change.getTxnNumber();
            if (!(this.metadata == null || sessionId.equals(this.metadata.get("mongoSessionId")) && txnNumber.equals(this.metadata.get("mongoTxnNumber")))) {
                this.metadata = null;
            }
            if (this.metadata == null) {
                Bson predicate = Filters.and((Bson[])new Bson[]{Filters.eq((String)"mongoSessionId", (Object)sessionId), Filters.eq((String)"mongoTxnNumber", (Object)txnNumber)});
                Document data = (Document)this.store.db.getCollection(AUDIT_METADATAS_COLLECTION).find(predicate).first();
                this.metadata = data != null ? new AuditMetadata((Map<String, Object>)data) : this.createMetadata(sessionId, txnNumber, change.getClusterTime());
            }
        }
    }

    private AuditMetadata createMetadata() {
        AuditMetadata metadata = this.newAuditMetadata();
        metadata.put("description", "<undocumented-changes>");
        metadata.setUTCTimestamp(LocalDateTime.now(ZoneOffset.UTC));
        this.store.db.getCollection(AUDIT_METADATAS_COLLECTION).insertOne((Object)metadata);
        return metadata;
    }

    private AuditMetadata createMetadata(Object sessionId, Object txnNumber, BsonTimestamp timestamp) {
        AuditMetadata metadata = this.newAuditMetadata();
        metadata.put("description", "<pre-existing-records>");
        metadata.put("mongoSessionId", sessionId);
        metadata.put("mongoTxnNumber", txnNumber);
        Instant instant = Instant.ofEpochSecond(timestamp.getTime());
        metadata.setUTCTimestamp(instant.atOffset(ZoneOffset.UTC).toLocalDateTime());
        this.store.db.getCollection(AUDIT_METADATAS_COLLECTION).insertOne((Object)metadata);
        return metadata;
    }

    public void stop() {
        if (this.thread == null) {
            return;
        }
        this.isTerminated.set(true);
        try {
            this.thread.join();
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
    }

    public AuditMetadata newAuditMetadata() {
        AuditMetadata meta = new AuditMetadata();
        meta.setAuditMetadataId(this.store.convertToDbId(UUID.randomUUID()));
        return meta;
    }

    public AuditMetadata populateAuditMetadata(ClientSession session, AuditMetadata auditMetadata) {
        if (auditMetadata == null) {
            auditMetadata = this.newAuditMetadata();
        }
        auditMetadata.setUTCTimestamp(LocalDateTime.now(ZoneId.of("UTC")));
        auditMetadata.put("mongoSessionId", session.getServerSession().getIdentifier().get((Object)"id"));
        auditMetadata.put("mongoTxnNumber", session.getServerSession().getTransactionNumber());
        return auditMetadata;
    }

    public PromptoDbId fetchLatestAuditMetadataId(PromptoDbId dbId) {
        Document data = (Document)this.store.db.getCollection(AUDIT_RECORDS_COLLECTION).find(Filters.eq((String)INSTANCE_DBID_FIELD_NAME, (Object)dbId.getValue())).projection(Projections.include((String[])new String[]{METADATA_DBID_FIELD_NAME})).sort(Sorts.orderBy((Bson[])new Bson[]{Sorts.descending((String[])new String[]{UTC_TIMESTAMP_FIELD_NAME})})).limit(1).first();
        return data == null ? null : this.store.convertToDbId(data.get((Object)METADATA_DBID_FIELD_NAME));
    }

    public PromptoList<PromptoDbId> fetchAllAuditMetadataIds(PromptoDbId dbId) {
        return (PromptoList)StreamSupport.stream(this.store.db.getCollection(AUDIT_RECORDS_COLLECTION).find(Filters.eq((String)INSTANCE_DBID_FIELD_NAME, (Object)dbId.getValue())).projection(Projections.include((String[])new String[]{METADATA_DBID_FIELD_NAME})).sort(Sorts.orderBy((Bson[])new Bson[]{Sorts.descending((String[])new String[]{UTC_TIMESTAMP_FIELD_NAME})})).spliterator(), false).map(data -> data.get((Object)METADATA_DBID_FIELD_NAME)).map(PromptoDbId::of).collect(PromptoList.collector());
    }

    public PromptoList<PromptoDbId> fetchDbIdsAffectedByAuditMetadataId(PromptoDbId dbId) {
        return (PromptoList)StreamSupport.stream(this.store.db.getCollection(AUDIT_RECORDS_COLLECTION).find(Filters.eq((String)METADATA_DBID_FIELD_NAME, (Object)dbId.getValue())).projection(Projections.include((String[])new String[]{INSTANCE_DBID_FIELD_NAME})).sort(Sorts.orderBy((Bson[])new Bson[]{Sorts.descending((String[])new String[]{UTC_TIMESTAMP_FIELD_NAME})})).spliterator(), false).map(data -> data.get((Object)INSTANCE_DBID_FIELD_NAME)).map(PromptoDbId::of).collect(PromptoList.collector());
    }

    public AuditRecord fetchLatestAuditRecord(PromptoDbId dbId) {
        Document data = (Document)this.store.db.getCollection(AUDIT_RECORDS_COLLECTION).find(Filters.eq((String)INSTANCE_DBID_FIELD_NAME, (Object)dbId.getValue())).sort(Sorts.orderBy((Bson[])new Bson[]{Sorts.descending((String[])new String[]{UTC_TIMESTAMP_FIELD_NAME})})).limit(1).first();
        return new AuditRecord(this.store, data);
    }

    public PromptoList<AuditRecord> fetchAllAuditRecords(PromptoDbId dbId) {
        return (PromptoList)StreamSupport.stream(this.store.db.getCollection(AUDIT_RECORDS_COLLECTION).find(Filters.eq((String)INSTANCE_DBID_FIELD_NAME, (Object)dbId.getValue())).sort(Sorts.orderBy((Bson[])new Bson[]{Sorts.descending((String[])new String[]{UTC_TIMESTAMP_FIELD_NAME})})).spliterator(), false).map(data -> new AuditRecord(this.store, (Document)data)).collect(PromptoList.collector());
    }

    public PromptoList<AuditRecord> fetchAuditRecordsMatching(Map<String, Object> auditPredicates, Map<String, Object> instancePredicates) {
        if ((auditPredicates == null ? 0 : auditPredicates.size()) + (instancePredicates == null ? 0 : instancePredicates.size()) == 0) {
            return new PromptoList(false);
        }
        List auditFilters = auditPredicates == null ? Collections.emptyList() : auditPredicates.entrySet().stream().map(e -> Filters.eq((String)((String)e.getKey()), (Object)PromptoToMongoConverter.convert(e.getValue()))).collect(Collectors.toList());
        List instanceFilters = instancePredicates == null ? Collections.emptyList() : instancePredicates.entrySet().stream().map(e -> Filters.eq((String)("instance." + (String)e.getKey()), (Object)PromptoToMongoConverter.convert(e.getValue()))).collect(Collectors.toList());
        List filters = Stream.concat(auditFilters.stream(), instanceFilters.stream()).collect(Collectors.toList());
        Bson filter = filters.size() > 1 ? Filters.and(filters) : (Bson)filters.get(0);
        return (PromptoList)StreamSupport.stream(this.store.db.getCollection(AUDIT_RECORDS_COLLECTION).find(filter).sort(Sorts.orderBy((Bson[])new Bson[]{Sorts.descending((String[])new String[]{UTC_TIMESTAMP_FIELD_NAME})})).spliterator(), false).map(data -> new AuditRecord(this.store, (Document)data)).collect(PromptoList.collector());
    }

    static class MongoToPromptoConverter {
        String table;
        PromptoDbId dbId;
        Stack<String> attributes = new Stack();

        MongoToPromptoConverter(String table, PromptoDbId dbId) {
            this.table = table;
            this.dbId = dbId;
        }

        public Object convert(Object value) {
            if (value instanceof Document) {
                return this.toPromptoDocument((Document)value);
            }
            if (value instanceof Date) {
                return MongoToPromptoConverter.toPromptoDateTime((Date)value);
            }
            if (value instanceof Binary) {
                PromptoBinary binary = MongoStore.binaryToPromptoBinary(value);
                String attribute = StreamSupport.stream(this.attributes.spliterator(), false).collect(Collectors.joining("."));
                binary.setSource(this.table, (Object)this.dbId, attribute);
                return binary;
            }
            return value;
        }

        private static PromptoDateTime toPromptoDateTime(Date value) {
            return new PromptoDateTime(new DateTime((Object)value, DateTimeZone.UTC));
        }

        private PromptoDocument<String, Object> toPromptoDocument(Document value) {
            PromptoDocument doc = new PromptoDocument((Map)value);
            for (String key : doc.keySet()) {
                this.attributes.push(key);
                doc.put((Object)key, this.convert(doc.get((Object)key)));
                this.attributes.pop();
            }
            return doc;
        }
    }

    static class PromptoToMongoConverter {
        PromptoToMongoConverter() {
        }

        public static Object convert(Object value) {
            if (value instanceof Enum) {
                return ((Enum)value).name();
            }
            return value;
        }
    }

    static class AuditMetadata
    extends Document
    implements IAuditMetadata {
        public AuditMetadata() {
        }

        public AuditMetadata(Map<String, Object> data) {
            super(data);
        }

        public <T> T get(String fieldName, Class<T> resultClass) {
            Object value;
            if (resultClass == LocalDateTime.class && (value = this.get(fieldName)) instanceof Date) {
                return (T)LocalDateTime.ofInstant(((Date)value).toInstant(), ZoneId.of("UTC"));
            }
            return (T)super.get((Object)fieldName, resultClass);
        }

        public PromptoDocument<String, Object> toDocument() {
            PromptoDbId dbId = PromptoDbId.of((Object)MongoDbIdConverter.toNative(this.get("auditMetadataId")));
            return (PromptoDocument)new MongoToPromptoConverter(MongoAuditor.AUDIT_METADATAS_COLLECTION, dbId).convert((Object)this);
        }
    }

    static class AuditRecord
    extends Document
    implements IAuditRecord {
        final MongoStore store;

        public AuditRecord(MongoStore store) {
            this.store = store;
        }

        public AuditRecord(MongoStore store, Document data) {
            super((Map)data);
            this.store = store;
        }

        public void setDbId(PromptoDbId id) {
            this.put("_id", id);
        }

        public PromptoDbId getDbId() {
            return this.store.convertToDbId(this.get("_id"));
        }

        public void setMetadataDbId(PromptoDbId id) {
            this.put(MongoAuditor.METADATA_DBID_FIELD_NAME, id);
        }

        public PromptoDbId getMetadataDbId() {
            return this.store.convertToDbId(this.get(MongoAuditor.METADATA_DBID_FIELD_NAME));
        }

        public void setUTCTimestamp(LocalDateTime timeStamp) {
            this.put(MongoAuditor.UTC_TIMESTAMP_FIELD_NAME, timeStamp);
        }

        public LocalDateTime getUTCTimestamp() {
            Object timeStamp = this.get(MongoAuditor.UTC_TIMESTAMP_FIELD_NAME);
            return timeStamp == null ? null : this.convertUTCTimestamp(timeStamp);
        }

        private LocalDateTime convertUTCTimestamp(Object timeStamp) {
            return null;
        }

        public void setInstanceDbId(PromptoDbId dbId) {
            this.put(MongoAuditor.INSTANCE_DBID_FIELD_NAME, dbId);
        }

        public PromptoDbId getInstanceDbId() {
            return this.store.convertToDbId(this.get(MongoAuditor.INSTANCE_DBID_FIELD_NAME));
        }

        public void setOperation(IAuditRecord.Operation operation) {
            this.put("operation", operation.name());
        }

        public IAuditRecord.Operation getOperation() {
            Object operation = this.get("operation");
            return operation == null ? null : IAuditRecord.Operation.valueOf((String)operation.toString());
        }

        public void setInstance(IStored stored) {
            this.put("instance", ((StoredDocument)stored).document);
        }

        public IStored getInstance() {
            Object instance = this.get("instance");
            return instance instanceof Document ? new StoredDocument(this.store, (Document)instance) : null;
        }

        public PromptoDocument<String, Object> toDocument() {
            return (PromptoDocument)new MongoToPromptoConverter(MongoAuditor.AUDIT_RECORDS_COLLECTION, this.getDbId()).convert((Object)this);
        }
    }
}

