package org.protelis.parser.validation;

import com.google.common.base.Objects;
import com.google.common.collect.ImmutableList;
import com.google.inject.Inject;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import org.eclipse.emf.common.notify.Notifier;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.common.types.JvmDeclaredType;
import org.eclipse.xtext.common.types.JvmFeature;
import org.eclipse.xtext.common.types.JvmType;
import org.eclipse.xtext.common.types.util.TypeReferences;
import org.eclipse.xtext.naming.IQualifiedNameConverter;
import org.eclipse.xtext.validation.Check;
import org.eclipse.xtext.xbase.interpreter.IEvaluationContext;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.protelis.parser.ProtelisExtensions;
import org.protelis.parser.protelis.Assignment;
import org.protelis.parser.protelis.Block;
import org.protelis.parser.protelis.BuiltinHoodOp;
import org.protelis.parser.protelis.Declaration;
import org.protelis.parser.protelis.Expression;
import org.protelis.parser.protelis.FunctionDef;
import org.protelis.parser.protelis.GenericHood;
import org.protelis.parser.protelis.Hood;
import org.protelis.parser.protelis.InvocationArguments;
import org.protelis.parser.protelis.It;
import org.protelis.parser.protelis.JavaImport;
import org.protelis.parser.protelis.KotlinStyleLambda;
import org.protelis.parser.protelis.Lambda;
import org.protelis.parser.protelis.LongLambda;
import org.protelis.parser.protelis.MethodCall;
import org.protelis.parser.protelis.OldLambda;
import org.protelis.parser.protelis.OldLongLambda;
import org.protelis.parser.protelis.OldShortLambda;
import org.protelis.parser.protelis.ProtelisModule;
import org.protelis.parser.protelis.ProtelisPackage;
import org.protelis.parser.protelis.ShortLambda;
import org.protelis.parser.protelis.Statement;
import org.protelis.parser.protelis.VarDef;
import org.protelis.parser.protelis.VarDefList;

/**
 * Custom validation rules.
 * 
 * see http://www.eclipse.org/Xtext/documentation.html#validation
 */
@SuppressWarnings("all")
public class ProtelisValidator extends AbstractProtelisValidator {
  public static final ImmutableList<Integer> MY_VERSION = ImmutableList.<Integer>of(Integer.valueOf(10), Integer.valueOf(0), Integer.valueOf(2));
  
  private static final EStructuralFeature FIRST_LINE = ProtelisPackage.Literals.PROTELIS_MODULE.getEStructuralFeature(ProtelisPackage.PROTELIS_MODULE__NAME);
  
  @Inject
  private TypeReferences references;
  
  @Inject
  private IEvaluationContext context;
  
  @Inject
  private IQualifiedNameConverter qualifiedNameConverter;
  
  private static final Iterable<String> AUTO_IMPORT = Collections.<String>unmodifiableSet(CollectionLiterals.<String>newHashSet("java.lang.Math", "java.lang.Double", "org.protelis.Builtins"));
  
  /**
   * Make sure that nobody defined the variable already:
   * 
   * other previous lets;
   * 
   * containing function;
   * 
   * containing lambda;
   * 
   * containing rep
   */
  @Check
  public void letNameDoesNotShadowArguments(final Declaration exp) {
    EObject parent = exp.eContainer();
    while ((parent != null)) {
      {
        if ((parent instanceof Block)) {
          final Function1<Statement, Boolean> _function = (Statement it) -> {
            return Boolean.valueOf((!Objects.equal(it, exp)));
          };
          final Function1<Statement, EObject> _function_1 = (Statement it) -> {
            EObject _xifexpression = null;
            if ((it instanceof Declaration)) {
              _xifexpression = ((Declaration)it).getName();
            } else {
              _xifexpression = it;
            }
            return _xifexpression;
          };
          final Function1<EObject, Boolean> _function_2 = (EObject it) -> {
            return Boolean.valueOf((it instanceof VarDef));
          };
          final Function1<EObject, VarDef> _function_3 = (EObject it) -> {
            return ((VarDef) it);
          };
          final Function1<VarDef, Boolean> _function_4 = (VarDef it) -> {
            String _name = it.getName();
            String _name_1 = exp.getName().getName();
            return Boolean.valueOf(Objects.equal(_name, _name_1));
          };
          final Consumer<VarDef> _function_5 = (VarDef it) -> {
            this.error(exp);
          };
          Optional.<VarDef>ofNullable(
            IterableExtensions.<VarDef>head(IterableExtensions.<VarDef>filter(IterableExtensions.<EObject, VarDef>map(IterableExtensions.<EObject>filter(IterableExtensions.<Statement, EObject>map(IterableExtensions.<Statement>takeWhile(((Block)parent).getStatements(), _function), _function_1), _function_2), _function_3), _function_4))).ifPresent(_function_5);
        }
        if ((parent instanceof FunctionDef)) {
          VarDefList _args = ((FunctionDef)parent).getArgs();
          boolean _tripleNotEquals = (_args != null);
          if (_tripleNotEquals) {
            final Function1<VarDef, String> _function_6 = (VarDef it) -> {
              return it.getName();
            };
            boolean _contains = ListExtensions.<VarDef, String>map(((FunctionDef)parent).getArgs().getArgs(), _function_6).contains(exp.getName());
            if (_contains) {
              this.error(exp);
            }
          }
        }
        if ((parent instanceof Lambda)) {
          List<VarDef> _switchResult = null;
          boolean _matched = false;
          if (parent instanceof OldLongLambda) {
            _matched=true;
            List<VarDef> _elvis = null;
            VarDefList _args_1 = ((OldLongLambda)parent).getArgs();
            EList<VarDef> _args_2 = null;
            if (_args_1!=null) {
              _args_2=_args_1.getArgs();
            }
            if (_args_2 != null) {
              _elvis = _args_2;
            } else {
              List<VarDef> _emptyList = CollectionLiterals.<VarDef>emptyList();
              _elvis = _emptyList;
            }
            _switchResult = _elvis;
          }
          if (!_matched) {
            if (parent instanceof OldShortLambda) {
              _matched=true;
              VarDef _singleArg = ((OldShortLambda)parent).getSingleArg();
              _switchResult = Collections.<VarDef>unmodifiableList(CollectionLiterals.<VarDef>newArrayList(_singleArg));
            }
          }
          if (!_matched) {
            if (parent instanceof ShortLambda) {
              _matched=true;
              _switchResult = CollectionLiterals.<VarDef>emptyList();
            }
          }
          if (!_matched) {
            if (parent instanceof LongLambda) {
              _matched=true;
              _switchResult = ((LongLambda)parent).getArgs().getArgs();
            }
          }
          final Iterable<VarDef> args = _switchResult;
          final Function1<VarDef, Boolean> _function_7 = (VarDef it) -> {
            return Boolean.valueOf(it.getName().equals(exp.getName()));
          };
          boolean _exists = IterableExtensions.<VarDef>exists(args, _function_7);
          if (_exists) {
            this.error(exp);
          }
        }
        parent = parent.eContainer();
      }
    }
  }
  
  private void error(final Declaration exp) {
    VarDef _name = exp.getName();
    String _plus = ("Variable " + _name);
    final String error = (_plus + " has already been defined in this context. Pick another name.");
    this.error(error, exp, null);
  }
  
  @Check
  public void lastElementOfBlockIsExpression(final Block block) {
    Statement _last = IterableExtensions.<Statement>last(block.getStatements());
    boolean _not = (!(_last instanceof Expression));
    if (_not) {
      this.error("The last statement in a returning block must be an expression", 
        IterableExtensions.<Statement>last(block.getStatements()), null);
    }
  }
  
  @Check
  public void noProtelisBuiltinsAvailable(final ProtelisModule module) {
    JvmType _findDeclaredType = this.references.findDeclaredType("org.protelis.Builtins", module);
    boolean _tripleEquals = (_findDeclaredType == null);
    if (_tripleEquals) {
      this.warning("No builtins available. Is the interpreter in the classpath?", module, 
        ProtelisPackage.Literals.PROTELIS_MODULE.getEStructuralFeature(ProtelisPackage.PROTELIS_MODULE__NAME));
    }
  }
  
  @Check
  public void warnOnOldLambda(final OldLambda lambda) {
    String _switchResult = null;
    boolean _matched = false;
    if (lambda instanceof OldShortLambda) {
      _matched=true;
      String _name = ((OldShortLambda)lambda).getSingleArg().getName();
      _switchResult = (_name + " -> ");
    }
    if (!_matched) {
      if (lambda instanceof OldLongLambda) {
        _matched=true;
        String _xifexpression = null;
        VarDefList _args = ((OldLongLambda)lambda).getArgs();
        boolean _tripleEquals = (_args == null);
        if (_tripleEquals) {
          _xifexpression = "";
        } else {
          final Function1<VarDef, CharSequence> _function = (VarDef it) -> {
            return it.getName();
          };
          _xifexpression = IterableExtensions.<VarDef>join(((OldLongLambda)lambda).getArgs().getArgs(), "", ", ", " -> ", _function);
        }
        _switchResult = _xifexpression;
      }
    }
    final String args = _switchResult;
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("This lambda could be rewritten as { ");
    _builder.append(args);
    _builder.append("<body> }");
    this.warning(_builder.toString(), null);
  }
  
  @Check
  public void deprecatedHoodOperation(final BuiltinHoodOp hood) {
    this.deprecatedHoodOperation(hood, ProtelisPackage.Literals.BUILTIN_HOOD_OP.getEStructuralFeature(ProtelisPackage.BUILTIN_HOOD_OP__NAME));
  }
  
  @Check
  public void deprecatedHoodOperation(final GenericHood hood) {
    this.deprecatedHoodOperation(hood, ProtelisPackage.Literals.HOOD.getEStructuralFeature(ProtelisPackage.HOOD__NAME));
  }
  
  public void deprecatedHoodOperation(final Hood hood, final EStructuralFeature feature) {
    this.warning("Hardcoded hood operations have been deprecated and will be removed in a future release, see https://github.com/Protelis/Protelis/issues/75", hood, feature);
  }
  
  @Check
  public void emptyStarImport(final JavaImport javaImport) {
    if ((javaImport.isWildcard() && IterableExtensions.isEmpty(ProtelisExtensions.callableEntities(javaImport.getImportedType())))) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("No callable members in ");
      String _simpleName = javaImport.getImportedType().getSimpleName();
      _builder.append(_simpleName);
      this.warning(_builder.toString(), ProtelisPackage.Literals.JAVA_IMPORT__IMPORTED_TYPE);
    }
  }
  
  @Check
  public void unresolvableMethodOrField(final JavaImport javaImport) {
    if (((!javaImport.isWildcard()) && IterableExtensions.isEmpty(ProtelisExtensions.callableEntitiesNamed(javaImport.getImportedType(), javaImport.getImportedMemberName())))) {
      StringConcatenation _builder = new StringConcatenation();
      String _importedMemberName = javaImport.getImportedMemberName();
      _builder.append(_importedMemberName);
      _builder.append(" is not a callable member of ");
      String _simpleName = javaImport.getImportedType().getSimpleName();
      _builder.append(_simpleName);
      this.error(_builder.toString(), 
        ProtelisPackage.Literals.JAVA_IMPORT__IMPORTED_MEMBER_NAME);
    }
  }
  
  private Iterable<JvmFeature> autoImports(final Notifier context) {
    final Function1<String, JvmDeclaredType> _function = (String it) -> {
      JvmType _findDeclaredType = this.references.findDeclaredType(it, context);
      return ((JvmDeclaredType) _findDeclaredType);
    };
    final Function1<JvmDeclaredType, Iterable<JvmFeature>> _function_1 = (JvmDeclaredType it) -> {
      return it.getAllFeatures();
    };
    return IterableExtensions.<JvmDeclaredType, JvmFeature>flatMap(IterableExtensions.<String, JvmDeclaredType>map(ProtelisValidator.AUTO_IMPORT, _function), _function_1);
  }
  
  @Check
  public void importOfAutomaticallyImportedClass(final JavaImport javaImport) {
    final Map<JvmFeature, JvmFeature> shadow = this.shadows(javaImport);
    boolean _isEmpty = shadow.isEmpty();
    boolean _not = (!_isEmpty);
    if (_not) {
      final BiConsumer<JvmFeature, JvmFeature> _function = (JvmFeature manual, JvmFeature auto) -> {
        StringConcatenation _builder = new StringConcatenation();
        String _qualifiedName = manual.getQualifiedName();
        _builder.append(_qualifiedName);
        _builder.append(" shadows automatically imported ");
        String _qualifiedName_1 = auto.getQualifiedName();
        _builder.append(_qualifiedName_1);
        this.info(_builder.toString(), 
          ProtelisPackage.Literals.JAVA_IMPORT__IMPORTED_TYPE);
      };
      shadow.forEach(_function);
    }
  }
  
  private Map<JvmFeature, JvmFeature> shadows(final JavaImport javaImport) {
    LinkedHashMap<JvmFeature, JvmFeature> _xblockexpression = null;
    {
      final Iterable<JvmFeature> automaticallyImported = this.autoImports(javaImport);
      final LinkedHashMap<JvmFeature, JvmFeature> shade = CollectionLiterals.<JvmFeature, JvmFeature>newLinkedHashMap();
      final Consumer<JvmFeature> _function = (JvmFeature manual) -> {
        final Function1<JvmFeature, Boolean> _function_1 = (JvmFeature it) -> {
          String _simpleName = it.getSimpleName();
          String _simpleName_1 = manual.getSimpleName();
          return Boolean.valueOf(Objects.equal(_simpleName, _simpleName_1));
        };
        final JvmFeature shadowed = IterableExtensions.<JvmFeature>findFirst(automaticallyImported, _function_1);
        if ((shadowed != null)) {
          shade.put(manual, shadowed);
        }
      };
      ProtelisExtensions.importedEntities(javaImport).forEach(_function);
      _xblockexpression = shade;
    }
    return _xblockexpression;
  }
  
  @Check
  public void invokeLambdaWithCorrectNumberOfArguments(final Expression invoke) {
    if (((((invoke.getName() == null) && (invoke.getElements().size() == 2)) && (invoke.getElements().get(0) instanceof Lambda)) && (invoke.getElements().get(1) instanceof InvocationArguments))) {
      EObject _get = invoke.getElements().get(0);
      final Lambda left = ((Lambda) _get);
      EObject _get_1 = invoke.getElements().get(1);
      final InvocationArguments args = ((InvocationArguments) _get_1);
      int _xifexpression = (int) 0;
      if (((args == null) || (args.getArgs() == null))) {
        _xifexpression = 0;
      } else {
        _xifexpression = args.getArgs().getArgs().size();
      }
      int _xifexpression_1 = (int) 0;
      KotlinStyleLambda _lastArg = args.getLastArg();
      boolean _tripleEquals = (_lastArg == null);
      if (_tripleEquals) {
        _xifexpression_1 = 0;
      } else {
        _xifexpression_1 = 1;
      }
      final int provided = (_xifexpression + _xifexpression_1);
      boolean _switchResult = false;
      boolean _matched = false;
      if (left instanceof OldShortLambda) {
        _matched=true;
        _switchResult = (provided == 1);
      }
      if (!_matched) {
        if (left instanceof OldLongLambda) {
          _matched=true;
          boolean _xifexpression_2 = false;
          VarDefList _args = ((OldLongLambda)left).getArgs();
          boolean _tripleEquals_1 = (_args == null);
          if (_tripleEquals_1) {
            _xifexpression_2 = (provided == 0);
          } else {
            int _size = ((OldLongLambda)left).getArgs().getArgs().size();
            _xifexpression_2 = (_size == provided);
          }
          _switchResult = _xifexpression_2;
        }
      }
      if (!_matched) {
        if (left instanceof ShortLambda) {
          _matched=true;
          _switchResult = ((provided == 0) || (provided == 1));
        }
      }
      if (!_matched) {
        if (left instanceof LongLambda) {
          _matched=true;
          int _size = ((LongLambda)left).getArgs().getArgs().size();
          _switchResult = (_size == provided);
        }
      }
      final boolean matches = _switchResult;
      if ((!matches)) {
        this.error("Arguments provided for invocation do not match lambda parameters", invoke, null);
      }
    }
  }
  
  private void error(final String message, final EObject target) {
    this.error(message, target, null);
  }
  
  private void warning(final String message, final EObject target) {
    this.warning(message, target, null);
  }
  
  @Check
  public void discourageDotApply(final Expression methodCall) {
    String _name = methodCall.getName();
    boolean _equals = Objects.equal(_name, ".");
    if (_equals) {
      EObject _get = methodCall.getElements().get(1);
      final MethodCall call = ((MethodCall) _get);
      String _name_1 = call.getName();
      boolean _equals_1 = Objects.equal(_name_1, "apply");
      if (_equals_1) {
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("<invokable>.apply(...) is discouraged, prefer direct invocation: <invokable>(...)");
        this.warning(_builder.toString(), call);
      }
    }
  }
  
  @Check
  public void reassignIsBadPractice(final Assignment assignment) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Reassigning variables such as ");
    String _name = assignment.getRefVar().getName();
    _builder.append(_name);
    _builder.append(" is discouraged and may be dropped in a future release of Protelis. Prefer a functional approach.");
    this.warning(_builder.toString(), assignment);
  }
  
  @Check
  public void uselessModule(final ProtelisModule module) {
    Block _program = module.getProgram();
    boolean _tripleEquals = (_program == null);
    if (_tripleEquals) {
      if (((module.getDefinitions() == null) || (IterableExtensions.<FunctionDef>findFirst(module.getDefinitions(), ((Function1<FunctionDef, Boolean>) (FunctionDef it) -> {
        return Boolean.valueOf(it.isPublic());
      })) == null))) {
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("Module ");
        String _elvis = null;
        String _name = module.getName();
        if (_name != null) {
          _elvis = _name;
        } else {
          _elvis = "anonymous";
        }
        _builder.append(_elvis);
        _builder.append(" is useless, it does not contain a program nor a public function");
        this.warning(_builder.toString(), module);
      }
    }
  }
  
  @Check
  public void appropriateUseOfIt(final It it) {
    int shortLambdaParents = 0;
    for (EObject parent = it.eContainer(); ((shortLambdaParents < 2) && (parent != null)); parent = parent.eContainer()) {
      int _shortLambdaParents = shortLambdaParents;
      int _xifexpression = (int) 0;
      if ((parent instanceof ShortLambda)) {
        _xifexpression = 1;
      } else {
        _xifexpression = 0;
      }
      shortLambdaParents = (_shortLambdaParents + _xifexpression);
    }
    switch (shortLambdaParents) {
      case 0:
        this.error("it can only be used inside short lambdas (i.e. { it + 1 })", it);
        break;
      case 2:
        this.error("Ambiguous use of it due to nested short lambdas: refactor with explicit names, e.g. rewrite { it + 1 } as { a -> a + 1 }", it);
        break;
    }
  }
  
  @Check
  public void functionCouldBeRewrittenAsSingleExpression(final FunctionDef function) {
    if (((function.getBody() != null) && (function.getBody().getStatements().size() == 1))) {
      final String name = function.getName();
      StringConcatenation _builder = new StringConcatenation();
      _builder.append(name);
      _builder.append(" (<params>) { <body> } has a single expression and could be rewritten as ");
      _builder.append(name);
      _builder.append(" (<params>) = <body>");
      this.info(_builder.toString(), function.getBody().getStatements().get(0), null);
    }
  }
  
  /**
   * See https://github.com/Protelis/Protelis/issues/245
   */
  @Check
  public void builtinVersionShouldBeCompatible(final ProtelisModule module) {
    final JvmType type = this.references.findDeclaredType("org.protelis.Builtins", module);
    if ((type instanceof JvmDeclaredType)) {
      Object _value = this.context.getValue(this.qualifiedNameConverter.toQualifiedName("org.protelis.Builtins"));
      final Class<?> builtinsResolvedClass = ((Class<?>) _value);
      final Field[] candidateFields = builtinsResolvedClass.getDeclaredFields();
      final Function1<Field, Boolean> _function = (Field it) -> {
        String _name = it.getName();
        return Boolean.valueOf(Objects.equal(_name, "MINIMUM_PARSER_VERSION"));
      };
      final Field min = IterableExtensions.<Field>findFirst(((Iterable<Field>)Conversions.doWrapArray(candidateFields)), _function);
      List<Integer> minVersion = this.versionFromStaticField(min);
      if ((minVersion == null)) {
        this.warning(
          "Builtins do not declare a minimum version, Protelis plugin / parser and interpreter versions may be mismatched", module, ProtelisValidator.FIRST_LINE);
      }
      List<Integer> _elvis = null;
      if (minVersion != null) {
        _elvis = minVersion;
      } else {
        _elvis = Collections.<Integer>unmodifiableList(CollectionLiterals.<Integer>newArrayList(Integer.valueOf(0), Integer.valueOf(0), Integer.valueOf(0)));
      }
      minVersion = _elvis;
      final Function1<Field, Boolean> _function_1 = (Field it) -> {
        String _name = it.getName();
        return Boolean.valueOf(Objects.equal(_name, "MAXIMUM_PARSER_VERSION"));
      };
      final Field max = IterableExtensions.<Field>findFirst(((Iterable<Field>)Conversions.doWrapArray(candidateFields)), _function_1);
      List<Integer> maxVersion = this.versionFromStaticField(max);
      if ((maxVersion == null)) {
        this.warning("Builtins do not declare a maximum version", module, ProtelisValidator.FIRST_LINE);
      }
      List<Integer> _elvis_1 = null;
      if (maxVersion != null) {
        _elvis_1 = maxVersion;
      } else {
        _elvis_1 = Collections.<Integer>unmodifiableList(CollectionLiterals.<Integer>newArrayList(Integer.valueOf(Integer.MAX_VALUE), Integer.valueOf(Integer.MAX_VALUE), Integer.valueOf(Integer.MAX_VALUE)));
      }
      maxVersion = _elvis_1;
      boolean _not = (!(this.versionMinorEqual(ProtelisValidator.MY_VERSION, maxVersion) && this.versionMinorEqual(minVersion, ProtelisValidator.MY_VERSION)));
      if (_not) {
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("Protelis plugin / parser and interpreter versions mismatch. Expected parser in range ");
        _builder.append(minVersion);
        _builder.append(" to ");
        _builder.append(maxVersion);
        _builder.append(", found version ");
        _builder.append(ProtelisValidator.MY_VERSION);
        this.warning(_builder.toString(), module, ProtelisValidator.FIRST_LINE);
      }
    }
  }
  
  private List<Integer> versionFromStaticField(final Field field) {
    try {
      if ((field == null)) {
        return null;
      } else {
        if ((Modifier.isStatic(field.getModifiers()) && List.class.isAssignableFrom(field.getType()))) {
          Object _get = field.get(null);
          final List<?> fieldValue = ((List<?>) _get);
          if (((fieldValue.size() == 3) && IterableExtensions.forall(fieldValue, ((Function1<Object, Boolean>) (Object it) -> {
            return Boolean.valueOf(Integer.class.isAssignableFrom(it.getClass()));
          })))) {
            return ((List<Integer>) fieldValue);
          }
        }
      }
      return null;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private boolean versionMinorEqual(final Iterable<Integer> a, final Iterable<Integer> b) {
    return (((IterableExtensions.isEmpty(a) && IterableExtensions.isEmpty(b)) || (IterableExtensions.<Integer>head(a).compareTo(IterableExtensions.<Integer>head(b)) < 0)) || (Objects.equal(IterableExtensions.<Integer>head(a), IterableExtensions.<Integer>head(b)) && this.versionMinorEqual(IterableExtensions.<Integer>drop(a, 1), IterableExtensions.<Integer>drop(b, 1))));
  }
}
