/*
 * Decompiled with CFR 0.152.
 */
package io.questdb.cutlass.text;

import io.questdb.cutlass.text.TextConfiguration;
import io.questdb.cutlass.text.TextMetadataDetector;
import io.questdb.cutlass.text.types.TypeAdapter;
import io.questdb.cutlass.text.types.TypeManager;
import io.questdb.log.Log;
import io.questdb.log.LogFactory;
import io.questdb.log.LogRecord;
import io.questdb.std.Mutable;
import io.questdb.std.ObjList;
import io.questdb.std.ObjectPool;
import io.questdb.std.Unsafe;
import io.questdb.std.str.DirectByteCharSequence;
import java.io.Closeable;

public class TextLexer
implements Closeable,
Mutable {
    private static final Log LOG = LogFactory.getLog(TextLexer.class);
    private final ObjList<DirectByteCharSequence> fields = new ObjList();
    private final ObjectPool<DirectByteCharSequence> csPool;
    private final TextMetadataDetector metadataDetector;
    private final int lineRollBufLimit;
    private CharSequence tableName;
    private boolean ignoreEolOnce;
    private long lineRollBufCur;
    private Listener textLexerListener;
    private long lastLineStart;
    private int lineRollBufLen;
    private long lineRollBufPtr;
    private boolean header;
    private long lastQuotePos = -1L;
    private long errorCount = 0L;
    private int lineCountLimit;
    private int fieldMax = -1;
    private int fieldIndex;
    private long lineCount;
    private boolean eol;
    private boolean useLineRollBuf = false;
    private boolean rollBufferUnusable = false;
    private byte columnDelimiter;
    private boolean inQuote;
    private boolean delayedOutQuote;
    private long fieldLo;
    private long fieldHi;
    private boolean skipLinesWithExtraValues;

    public TextLexer(TextConfiguration textConfiguration, TypeManager typeManager) {
        this.metadataDetector = new TextMetadataDetector(typeManager, textConfiguration);
        this.csPool = new ObjectPool<DirectByteCharSequence>(DirectByteCharSequence.FACTORY, textConfiguration.getTextLexerStringPoolCapacity());
        this.lineRollBufLen = textConfiguration.getRollBufferSize();
        this.lineRollBufLimit = textConfiguration.getRollBufferLimit();
        this.lineRollBufPtr = Unsafe.malloc(this.lineRollBufLen);
    }

    public void analyseStructure(long lo, long hi, int lineCountLimit, boolean forceHeader, ObjList<CharSequence> names, ObjList<TypeAdapter> types) {
        this.metadataDetector.of(names, types, forceHeader);
        this.parse(lo, hi, lineCountLimit, this.metadataDetector);
        this.metadataDetector.evaluateResults(this.lineCount, this.errorCount);
        this.restart(this.isHeaderDetected());
    }

    @Override
    public final void clear() {
        this.restart(false);
        this.fields.clear();
        this.csPool.clear();
        this.metadataDetector.clear();
        this.errorCount = 0L;
        this.fieldMax = -1;
    }

    public long getErrorCount() {
        return this.errorCount;
    }

    @Override
    public void close() {
        if (this.lineRollBufPtr != 0L) {
            Unsafe.free(this.lineRollBufPtr, this.lineRollBufLen);
            this.lineRollBufPtr = 0L;
        }
        this.metadataDetector.close();
    }

    public long getLineCount() {
        return this.lineCount;
    }

    public void of(byte columnDelimiter) {
        this.clear();
        this.columnDelimiter = columnDelimiter;
    }

    public void parse(long lo, long hi, int lineCountLimit, Listener textLexerListener) {
        this.textLexerListener = textLexerListener;
        this.fieldHi = this.useLineRollBuf ? this.lineRollBufCur : (this.fieldLo = lo);
        this.lineCountLimit = lineCountLimit;
        this.parse(lo, hi);
    }

    public void parseLast() {
        if (this.useLineRollBuf) {
            if (this.inQuote && this.lastQuotePos < this.fieldHi) {
                ++this.errorCount;
                LOG.info().$("quote is missing [table=").$(this.tableName).$(']').$();
            } else {
                ++this.fieldHi;
                this.stashField(this.fieldIndex);
                this.triggerLine(0L);
            }
        }
    }

    public final void restart(boolean header) {
        this.fieldLo = 0L;
        this.eol = false;
        this.fieldIndex = 0;
        this.fieldMax = -1;
        this.inQuote = false;
        this.delayedOutQuote = false;
        this.lineCount = 0L;
        this.lineRollBufCur = this.lineRollBufPtr;
        this.useLineRollBuf = false;
        this.rollBufferUnusable = false;
        this.header = header;
        this.fields.clear();
        this.csPool.clear();
    }

    private void addField() {
        this.fields.add(this.csPool.next());
        ++this.fieldMax;
    }

    private void checkEol(long lo) {
        if (this.eol) {
            this.uneol(lo);
        }
    }

    private void clearRollBuffer(long ptr) {
        this.useLineRollBuf = false;
        this.lineRollBufCur = this.lineRollBufPtr;
        this.fieldLo = this.fieldHi = ptr;
    }

    private void eol(long ptr, byte c) {
        if (c == 10 || c == 13) {
            this.eol = true;
            this.rollBufferUnusable = false;
            this.clearRollBuffer(ptr);
            this.fieldIndex = 0;
            ++this.lineCount;
        }
    }

    ObjList<CharSequence> getColumnNames() {
        return this.metadataDetector.getColumnNames();
    }

    ObjList<TypeAdapter> getColumnTypes() {
        return this.metadataDetector.getColumnTypes();
    }

    private boolean growRollBuf(int requiredLength, boolean updateFields) {
        if (requiredLength > this.lineRollBufLimit) {
            LOG.info().$("too long [table=").$(this.tableName).$(", line=").$(this.lineCount).$(", requiredLen=").$(requiredLength).$(", rollLimit=").$(this.lineRollBufLimit).$(']').$();
            ++this.errorCount;
            this.rollBufferUnusable = true;
            return false;
        }
        int len = Math.min(this.lineRollBufLimit, requiredLength << 1);
        LOG.info().$("resizing ").$(this.lineRollBufLen).$(" -> ").$(len).$(" [table=").$(this.tableName).$(']').$();
        long p = Unsafe.malloc(len);
        long l = this.lineRollBufCur - this.lineRollBufPtr;
        if (l > 0L) {
            Unsafe.getUnsafe().copyMemory(this.lineRollBufPtr, p, l);
        }
        Unsafe.free(this.lineRollBufPtr, this.lineRollBufLen);
        if (updateFields) {
            this.shift(this.lineRollBufPtr - p);
        }
        this.lineRollBufCur = p + l;
        this.lineRollBufPtr = p;
        this.lineRollBufLen = len;
        return true;
    }

    private void growRollBufAndPut(byte c) {
        if (this.growRollBuf(this.lineRollBufLen + 1, true)) {
            Unsafe.getUnsafe().putByte(this.lineRollBufCur++, c);
        }
    }

    public boolean isSkipLinesWithExtraValues() {
        return this.skipLinesWithExtraValues;
    }

    public void setSkipLinesWithExtraValues(boolean skipLinesWithExtraValues) {
        this.skipLinesWithExtraValues = skipLinesWithExtraValues;
    }

    private void ignoreEolOnce() {
        this.eol = true;
        this.fieldIndex = 0;
        this.ignoreEolOnce = false;
    }

    boolean isHeaderDetected() {
        return this.metadataDetector.isHeader();
    }

    private void onColumnDelimiter(long lo) {
        this.checkEol(lo);
        if (this.inQuote || this.ignoreEolOnce) {
            return;
        }
        this.stashField(this.fieldIndex++);
    }

    private void onLineEnd(long ptr) throws LineLimitException {
        if (this.inQuote) {
            return;
        }
        if (this.eol) {
            this.fieldLo = this.fieldHi;
            return;
        }
        this.stashField(this.fieldIndex);
        if (this.ignoreEolOnce) {
            this.ignoreEolOnce();
            return;
        }
        this.triggerLine(ptr);
        if (this.lineCount > (long)this.lineCountLimit) {
            throw LineLimitException.INSTANCE;
        }
    }

    private void onQuote() {
        if (this.inQuote) {
            this.delayedOutQuote = !this.delayedOutQuote;
            this.lastQuotePos = this.fieldHi;
        } else if (this.fieldHi - this.fieldLo == 1L) {
            this.inQuote = true;
            this.fieldLo = this.fieldHi;
        }
    }

    private void parse(long lo, long hi) {
        long ptr = lo;
        try {
            while (ptr < hi) {
                byte c = Unsafe.getUnsafe().getByte(ptr++);
                if (this.rollBufferUnusable) {
                    this.eol(ptr, c);
                    continue;
                }
                if (this.useLineRollBuf) {
                    this.putToRollBuf(c);
                    if (this.rollBufferUnusable) continue;
                }
                ++this.fieldHi;
                if (this.delayedOutQuote && c != 34) {
                    this.delayedOutQuote = false;
                    this.inQuote = false;
                }
                if (c == this.columnDelimiter) {
                    this.onColumnDelimiter(lo);
                    continue;
                }
                if (c == 34) {
                    this.onQuote();
                    continue;
                }
                if (c == 10 || c == 13) {
                    this.onLineEnd(ptr);
                    continue;
                }
                this.checkEol(lo);
            }
        }
        catch (LineLimitException lineLimitException) {
            // empty catch block
        }
        if (this.useLineRollBuf) {
            return;
        }
        if (this.eol) {
            this.fieldLo = 0L;
        } else {
            this.rollLine(lo, hi);
            this.useLineRollBuf = true;
        }
    }

    private void putToRollBuf(byte c) {
        if (this.lineRollBufCur - this.lineRollBufPtr == (long)this.lineRollBufLen) {
            this.growRollBufAndPut(c);
        } else {
            Unsafe.getUnsafe().putByte(this.lineRollBufCur++, c);
        }
    }

    private void rollLine(long lo, long hi) {
        int l = (int)(hi - lo - this.lastLineStart);
        if (l < this.lineRollBufLen || this.growRollBuf(l, false)) {
            assert (lo + this.lastLineStart + (long)l <= hi);
            Unsafe.getUnsafe().copyMemory(lo + this.lastLineStart, this.lineRollBufPtr, l);
            this.lineRollBufCur = this.lineRollBufPtr + (long)l;
            this.shift(lo + this.lastLineStart - this.lineRollBufPtr);
        }
    }

    void setTableName(CharSequence tableName) {
        this.tableName = tableName;
        this.metadataDetector.setTableName(tableName);
    }

    private void shift(long d) {
        for (int i = 0; i < this.fieldIndex; ++i) {
            this.fields.getQuick(i).lshift(d);
        }
        this.fieldLo -= d;
        this.fieldHi -= d;
        if (this.lastQuotePos > -1L) {
            this.lastQuotePos -= d;
        }
    }

    private void stashField(int fieldIndex) {
        if (this.lineCount == 0L && fieldIndex >= this.fields.size()) {
            this.addField();
        }
        if (fieldIndex > this.fieldMax) {
            LogRecord logRecord = LOG.error().$("extra fields [table=").$(this.tableName).$(", fieldIndex=").$(fieldIndex).$(", fieldMax=").$(this.fieldMax).$("]\n\t").$(this.lineCount).$(" -> ");
            int n = this.fields.size();
            for (int i = 0; i < n; ++i) {
                if (i > 0) {
                    logRecord.$(',');
                }
                logRecord.$(this.fields.getQuick(i));
            }
            logRecord.$(" ...").$();
            if (this.skipLinesWithExtraValues) {
                ++this.errorCount;
                this.ignoreEolOnce = true;
                this.fieldIndex = 0;
            } else {
                if (this.lastQuotePos > -1L) {
                    this.lastQuotePos = -1L;
                }
                this.fieldLo = this.fieldHi;
            }
            return;
        }
        if (this.lastQuotePos > -1L) {
            this.fields.getQuick(fieldIndex).of(this.fieldLo, this.lastQuotePos - 1L);
            this.lastQuotePos = -1L;
        } else {
            this.fields.getQuick(fieldIndex).of(this.fieldLo, this.fieldHi - 1L);
        }
        this.fieldLo = this.fieldHi;
    }

    private void triggerLine(long ptr) {
        this.eol = true;
        this.fieldIndex = 0;
        if (this.useLineRollBuf) {
            this.clearRollBuffer(ptr);
        }
        if (this.header) {
            this.header = false;
            return;
        }
        this.textLexerListener.onFields(this.lineCount++, this.fields, this.fieldMax + 1);
    }

    private void uneol(long lo) {
        this.eol = false;
        this.lastLineStart = this.fieldLo - lo;
    }

    private static final class LineLimitException
    extends Exception {
        private static final LineLimitException INSTANCE = new LineLimitException();

        private LineLimitException() {
        }
    }

    @FunctionalInterface
    public static interface Listener {
        public void onFields(long var1, ObjList<DirectByteCharSequence> var3, int var4);
    }
}

