/*
 * Decompiled with CFR 0.152.
 */
package com.questdb.net;

import com.questdb.common.JournalRuntimeException;
import com.questdb.log.Log;
import com.questdb.log.LogFactory;
import com.questdb.net.SslConfig;
import com.questdb.std.ByteBuffers;
import com.questdb.std.ex.JournalNetworkException;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.ByteChannel;
import java.nio.channels.SocketChannel;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLEngine;
import javax.net.ssl.SSLEngineResult;
import javax.net.ssl.SSLException;
import javax.net.ssl.SSLSession;

public class SecureSocketChannel
implements ByteChannel {
    private static final Log LOG = LogFactory.getLog(SecureSocketChannel.class);
    private final SocketChannel socketChannel;
    private final SSLEngine engine;
    private final ByteBuffer inBuf;
    private final ByteBuffer outBuf;
    private final int sslDataLimit;
    private final boolean client;
    private final ByteBuffer swapBuf;
    private boolean inData = false;
    private SSLEngineResult.HandshakeStatus handshakeStatus = SSLEngineResult.HandshakeStatus.NEED_WRAP;
    private boolean fillInBuf = true;

    public SecureSocketChannel(SocketChannel socketChannel, SslConfig sslConfig) {
        this.socketChannel = socketChannel;
        SSLContext sslc = sslConfig.getSslContext();
        this.engine = sslc.createSSLEngine();
        this.engine.setEnableSessionCreation(true);
        this.engine.setUseClientMode(sslConfig.isClient());
        this.engine.setNeedClientAuth(sslConfig.isRequireClientAuth());
        this.client = sslConfig.isClient();
        SSLSession session = this.engine.getSession();
        this.sslDataLimit = session.getApplicationBufferSize();
        this.inBuf = ByteBuffer.allocateDirect(session.getPacketBufferSize()).order(ByteOrder.LITTLE_ENDIAN);
        this.outBuf = ByteBuffer.allocateDirect(session.getPacketBufferSize()).order(ByteOrder.LITTLE_ENDIAN);
        this.swapBuf = ByteBuffer.allocateDirect(this.sslDataLimit * 2).order(ByteOrder.LITTLE_ENDIAN);
    }

    @Override
    public boolean isOpen() {
        return this.socketChannel.isOpen();
    }

    @Override
    public void close() throws IOException {
        this.socketChannel.close();
        ByteBuffers.release(this.inBuf);
        ByteBuffers.release(this.outBuf);
        ByteBuffers.release(this.swapBuf);
        if (this.engine.isOutboundDone()) {
            this.engine.closeOutbound();
        }
        while (!this.engine.isInboundDone()) {
            try {
                this.engine.closeInbound();
            }
            catch (SSLException sSLException) {}
        }
    }

    @Override
    public int read(ByteBuffer dst) throws IOException {
        int limit;
        if (this.handshakeStatus != SSLEngineResult.HandshakeStatus.FINISHED) {
            this.handshake();
        }
        int p = dst.position();
        while ((limit = dst.remaining()) != 0) {
            if (this.swapBuf.hasRemaining()) {
                ByteBuffers.copy(this.swapBuf, dst);
                continue;
            }
            if (this.fillInBuf) {
                this.inBuf.clear();
                int result = this.socketChannel.read(this.inBuf);
                if (result == -1) {
                    throw new IOException("Did not expect -1 from socket channel");
                }
                if (result == 0) {
                    throw new IOException("Blocking connection must not return 0");
                }
                this.inBuf.flip();
            }
            if (limit < this.sslDataLimit) {
                this.swapBuf.clear();
                this.fillInBuf = this.unwrap(this.swapBuf);
                this.swapBuf.flip();
                ByteBuffers.copy(this.swapBuf, dst);
                continue;
            }
            this.fillInBuf = this.unwrap(dst);
        }
        return dst.position() - p;
    }

    @Override
    public int write(ByteBuffer src) throws IOException {
        if (this.handshakeStatus != SSLEngineResult.HandshakeStatus.FINISHED) {
            this.handshake();
        }
        int count = src.remaining();
        while (src.hasRemaining()) {
            this.outBuf.clear();
            SSLEngineResult result = this.engine.wrap(src, this.outBuf);
            if (result.getStatus() != SSLEngineResult.Status.OK) {
                throw new IOException("Expected OK, got: " + (Object)((Object)result.getStatus()));
            }
            this.outBuf.flip();
            try {
                ByteBuffers.copy(this.outBuf, this.socketChannel);
            }
            catch (JournalNetworkException e) {
                throw new IOException(e);
            }
        }
        return count;
    }

    private void closureOnException() throws IOException {
        SSLEngineResult sslEngineResult;
        this.swapBuf.position(0);
        this.swapBuf.limit(0);
        do {
            this.outBuf.clear();
            sslEngineResult = this.engine.wrap(this.swapBuf, this.outBuf);
            this.outBuf.flip();
            this.socketChannel.write(this.outBuf);
        } while (sslEngineResult.getStatus() != SSLEngineResult.Status.CLOSED && !this.engine.isInboundDone());
        this.engine.closeOutbound();
    }

    private void handshake() throws IOException {
        if (this.handshakeStatus == SSLEngineResult.HandshakeStatus.FINISHED) {
            return;
        }
        this.engine.beginHandshake();
        block16: while (this.handshakeStatus != SSLEngineResult.HandshakeStatus.FINISHED) {
            switch (this.handshakeStatus) {
                case NOT_HANDSHAKING: {
                    throw new IOException("Not handshaking");
                }
                case NEED_WRAP: {
                    this.outBuf.clear();
                    this.swapBuf.clear();
                    try {
                        this.handshakeStatus = this.engine.wrap(this.swapBuf, this.outBuf).getHandshakeStatus();
                    }
                    catch (SSLException e) {
                        LOG.error().$("Server SSL handshake failed: ").$(e.getMessage()).$();
                        this.closureOnException();
                        this.socketChannel.close();
                        throw e;
                    }
                    this.outBuf.flip();
                    this.socketChannel.write(this.outBuf);
                    continue block16;
                }
                case NEED_UNWRAP: {
                    if (!this.inData || !this.inBuf.hasRemaining()) {
                        this.inBuf.clear();
                        this.socketChannel.read(this.inBuf);
                        this.inBuf.flip();
                        this.inData = true;
                    }
                    try {
                        SSLEngineResult res = this.engine.unwrap(this.inBuf, this.swapBuf);
                        this.handshakeStatus = res.getHandshakeStatus();
                        switch (res.getStatus()) {
                            case BUFFER_UNDERFLOW: {
                                this.inBuf.compact();
                                this.socketChannel.read(this.inBuf);
                                this.inBuf.flip();
                                break;
                            }
                            case BUFFER_OVERFLOW: {
                                throw new IOException("Did not expect OVERFLOW here");
                            }
                            case OK: {
                                break;
                            }
                            case CLOSED: {
                                throw new IOException("Did not expect CLOSED");
                            }
                        }
                        continue block16;
                    }
                    catch (SSLException e) {
                        LOG.error().$("Client SSL handshake failed: ").$(e.getMessage()).$();
                        this.handshakeStatus = SSLEngineResult.HandshakeStatus.FINISHED;
                        this.socketChannel.close();
                        throw e;
                    }
                }
                case NEED_TASK: {
                    Runnable task;
                    while ((task = this.engine.getDelegatedTask()) != null) {
                        task.run();
                    }
                    this.handshakeStatus = this.engine.getHandshakeStatus();
                    continue block16;
                }
            }
            throw new JournalRuntimeException("Unknown handshake status: %s", new Object[]{this.handshakeStatus});
        }
        this.inBuf.clear();
        this.swapBuf.position(this.swapBuf.limit());
        LOG.info().$("Handshake SSL complete: ").$(this.client ? "CLIENT" : "SERVER").$();
    }

    private boolean unwrap(ByteBuffer dst) throws IOException {
        while (this.inBuf.hasRemaining()) {
            SSLEngineResult.Status status = this.engine.unwrap(this.inBuf, dst).getStatus();
            switch (status) {
                case BUFFER_UNDERFLOW: {
                    this.inBuf.compact();
                    this.socketChannel.read(this.inBuf);
                    this.inBuf.flip();
                    break;
                }
                case BUFFER_OVERFLOW: {
                    return false;
                }
                case OK: {
                    break;
                }
                case CLOSED: {
                    throw new IOException("Did not expect CLOSED");
                }
            }
        }
        return true;
    }
}

