/*******************************************************************************
 *    ___                  _   ____  ____
 *   / _ \ _   _  ___  ___| |_|  _ \| __ )
 *  | | | | | | |/ _ \/ __| __| | | |  _ \
 *  | |_| | |_| |  __/\__ \ |_| |_| | |_) |
 *   \__\_\\__,_|\___||___/\__|____/|____/
 *
 * Copyright (C) 2014-2017 Appsicle
 *
 * This program is free software: you can redistribute it and/or  modify
 * it under the terms of the GNU Affero General Public License, version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 ******************************************************************************/

package com.questdb.std.str;

import com.questdb.common.JournalRuntimeException;
import com.questdb.std.ByteBuffers;
import com.questdb.std.Unsafe;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;

public class FileSink extends AbstractCharSink implements Closeable {

    private final RandomAccessFile raf;
    private final FileChannel channel;
    private ByteBuffer buffer;
    private long pos;
    private long addr;
    private long limit;

    public FileSink(File file) throws IOException {
        this.raf = new RandomAccessFile(file, "rw");
        this.channel = raf.getChannel();
        this.pos = 0L;
        this.addr = this.limit = 0;
    }

    @Override
    public void close() throws IOException {
        ByteBuffers.release(buffer);
        if (addr > 0 && pos > 0) {
            channel.truncate(pos - (limit - addr));
        }
        channel.close();
        raf.close();
    }

    @Override
    public CharSink put(CharSequence cs) {
        if (cs == null) {
            return this;
        }

        for (int i = 0, l = cs.length(); i < l; i++) {
            if (addr == limit) {
                map();
            }
            Unsafe.getUnsafe().putByte(addr++, (byte) cs.charAt(i));
        }
        return this;
    }

    @Override
    public CharSink put(char c) {
        if (addr == limit) {
            map();
        }
        Unsafe.getUnsafe().putByte(addr++, (byte) c);
        return this;
    }

    private void map() {
        if (buffer != null) {
            ByteBuffers.release(buffer);
        }
        try {
            this.buffer = channel.map(FileChannel.MapMode.READ_WRITE, this.pos, ByteBuffers.getMaxMappedBufferSize(Long.MAX_VALUE));
            pos += buffer.limit();
            addr = ByteBuffers.getAddress(buffer);
            limit = addr + buffer.remaining();
        } catch (IOException e) {
            throw new JournalRuntimeException(e);
        }
    }
}
