package org.randombits.confluence.metadata.indexing.graph;

import com.atlassian.confluence.event.events.cluster.ClusterEventWrapper;
import com.atlassian.event.Event;
import com.atlassian.event.api.EventListener;
import com.atlassian.event.api.EventPublisher;
import com.tinkerpop.blueprints.Direction;
import com.tinkerpop.blueprints.Vertex;
import com.tinkerpop.blueprints.impls.orient.OrientGraph;
import org.randombits.confluence.metadata.event.MetadataUpdatedEvent;
import org.randombits.confluence.metadata.indexing.FieldNotFoundException;
import org.randombits.confluence.metadata.indexing.IndexManager;
import org.randombits.confluence.metadata.indexing.graph.vertex.Content;
import org.randombits.confluence.metadata.indexing.graph.vertex.KeyValueField;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.Map;

/**
 * Default implementation of {@link org.randombits.confluence.metadata.indexing.IndexManager}.
 *
 * @author kaifung
 * @since 7.0.0.20150209
 */
@Component
public class DefaultIndexManager implements IndexManager, DisposableBean {

    private static final Logger log = LoggerFactory.getLogger(IndexManager.class);

    private final EventPublisher eventPublisher;

    private DatabaseBean databaseBean;

    public DefaultIndexManager(EventPublisher eventPublisher) {
        this.eventPublisher = eventPublisher;
        this.eventPublisher.register(this);
    }

    @Override
    public <T> T query(String contentId, String fieldName, Class<? extends T> clazz) throws IllegalArgumentException, FieldNotFoundException {
        if (!String.class.equals(clazz)) {
            throw new IllegalArgumentException("Unable to query index with invalid type : " + clazz);
        }

        // Retrieve Contents.
        for (Vertex content : databaseBean.getGraph().getVertices(Content.CONTENT_ID.getPrefixedPropertyKey(), contentId)) {
            // Look up for the specified field name.
            for (Vertex vertex : content.query().direction(Direction.OUT).labels(Content.EDGE_CONTAINS).vertices()) {
                if (fieldName.equals(vertex.getProperty(KeyValueField.NAME.toCanonicalValue()))) {
                    return vertex.getProperty(KeyValueField.VALUE.toCanonicalValue());
                }
            }
        }

        throw new FieldNotFoundException("Unable to find field with name : " + fieldName);
    }

    @Override
    public void buildIndex(String contentId, Map<String, Object> metadata) {
        Vertex content = replaceContentVertex(contentId);
        for (Map.Entry<String, Object> entry : metadata.entrySet()) {
            if (entry.getValue() instanceof String) {
                indexStringField(content, entry.getKey(), (String) entry.getValue());
            }
        }
    }

    private Vertex replaceContentVertex(String contentId) {
        OrientGraph orientGraph = databaseBean.getGraph();

        if(orientGraph.countVertices() != 0 ) {
            for (Vertex content : orientGraph.getVertices(Content.CONTENT_ID.getPrefixedPropertyKey(), contentId)) {
                orientGraph.removeVertex(content);
            }
        }

        return orientGraph.addVertex("class:" + Content.CLASS_NAME, Content.CONTENT_ID.toCanonicalValue(), contentId);
    }

    private void indexStringField(Vertex outVertex, String fieldName, String value) {
        Vertex inVertex = databaseBean.getGraph().addVertex("class:" + KeyValueField.CLASS_NAME,
            KeyValueField.NAME.toCanonicalValue(), fieldName,
            KeyValueField.VALUE.toCanonicalValue(), value
        );
        // set relationship
        databaseBean.getGraph().addEdge(null, outVertex, inVertex, Content.EDGE_CONTAINS);
    }

    /**
     * Listen to {@link org.randombits.confluence.metadata.event.MetadataUpdatedEvent}.
     *
     * @param metadataUpdatedEvent
     */
    @EventListener
    public void onEvent(MetadataUpdatedEvent metadataUpdatedEvent) {
        buildIndex(
            metadataUpdatedEvent.getMetadataStorage().getContent().getIdAsString(),
            metadataUpdatedEvent.getMetadataStorage().getBaseMap()
        );
    }

    /**
     * Listen to events from other nodes in clustered environment.
     *
     * @param clusterEventWrapper
     */
    @EventListener
    public void onEvent(ClusterEventWrapper clusterEventWrapper) {
        Event event = clusterEventWrapper.getEvent();
        if (event instanceof MetadataUpdatedEvent) {
            onEvent((MetadataUpdatedEvent) event);
        }
    }

    @Autowired
    public void setDatabaseBean(DatabaseBean databaseBean) {
        this.databaseBean = databaseBean;
    }

    @Override
    public void destroy() throws Exception {
        eventPublisher.unregister(this);
    }
}