/*
 * Decompiled with CFR 0.152.
 */
package org.rcsb.strucmotif.io;

import java.io.BufferedInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UncheckedIOException;
import java.nio.file.CopyOption;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;
import org.rcsb.strucmotif.config.InvertedIndexBackend;
import org.rcsb.strucmotif.config.StrucmotifConfig;
import org.rcsb.strucmotif.domain.bucket.Bucket;
import org.rcsb.strucmotif.domain.bucket.InvertedIndexBucket;
import org.rcsb.strucmotif.domain.bucket.ResiduePairIdentifierBucket;
import org.rcsb.strucmotif.domain.motif.AngleType;
import org.rcsb.strucmotif.domain.motif.DistanceType;
import org.rcsb.strucmotif.domain.motif.ResiduePairDescriptor;
import org.rcsb.strucmotif.domain.motif.ResiduePairIdentifier;
import org.rcsb.strucmotif.domain.structure.ResidueType;
import org.rcsb.strucmotif.io.InvertedIndex;
import org.rcsb.strucmotif.io.codec.BucketCodec;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

@Service
public class InvertedIndexImpl
implements InvertedIndex {
    private static final Logger logger = LoggerFactory.getLogger(InvertedIndexImpl.class);
    private static final Map<String, ResidueType> OLC_LOOKUP = Stream.of(ResidueType.values()).collect(Collectors.toMap(ResidueType::getInternalCode, Function.identity()));
    private static final int BUFFER_SIZE = 65536;
    private final Path basePath;
    private final boolean gzipped;
    private final String extension;
    private final BucketCodec bucketCodec;
    private final Pattern tempExtension;
    private boolean paths;

    public InvertedIndexImpl(StrucmotifConfig strucmotifConfig) {
        this.basePath = Paths.get(strucmotifConfig.getRootPath(), new String[0]).resolve("index");
        this.gzipped = strucmotifConfig.isInvertedIndexGzip();
        InvertedIndexBackend backend = strucmotifConfig.getInvertedIndexBackend();
        this.bucketCodec = backend.getBucketCodec();
        this.extension = backend.getExtension() + (this.gzipped ? ".gz" : "");
        logger.info("Index files will {}be gzipped - extension: {}", (Object)(this.gzipped ? "" : "not "), (Object)this.extension);
        this.tempExtension = Pattern.compile(".*" + this.extension.replace(".", "\\.") + "\\.[0-9]+$");
        this.paths = false;
    }

    @Override
    public void insert(ResiduePairDescriptor residuePairDescriptor, Bucket bucket, int batchId) {
        if (!this.paths) {
            this.ensureDirectoriesExist();
            this.paths = true;
        }
        try {
            Path tmpPath = this.getPath(residuePairDescriptor, batchId);
            try (ByteArrayOutputStream outputStream = this.bucketCodec.encode(bucket);){
                this.write(tmpPath, outputStream);
            }
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    @Override
    public void commit() {
        logger.info("Committing temporary files to index");
        try {
            ConcurrentHashMap toMerge = new ConcurrentHashMap();
            try (Stream<Path> paths = this.temporaryFiles();){
                AtomicInteger counter = new AtomicInteger();
                paths.peek(p -> this.progress(counter, 10000, "{} files scanned")).forEach(p -> {
                    String filename = p.getFileName().toString();
                    Path persistentPath = p.resolveSibling(filename.substring(0, filename.lastIndexOf(".")));
                    Set bin = toMerge.computeIfAbsent(persistentPath, e -> Collections.synchronizedSet(new HashSet()));
                    bin.add(persistentPath.resolveSibling((Path)p));
                });
            }
            logger.info("Merging {} bins", (Object)toMerge.size());
            toMerge.entrySet().parallelStream().forEach(entry -> {
                Path destination = (Path)entry.getKey();
                Set sources = (Set)entry.getValue();
                try {
                    HashMap<Integer, Collection<ResiduePairIdentifier>> merged = new HashMap<Integer, Collection<ResiduePairIdentifier>>();
                    if (Files.exists(destination, new LinkOption[0])) {
                        try (InputStream inputStream = this.getInputStream(destination);){
                            this.addAll(merged, this.bucketCodec.decode(inputStream), null);
                        }
                    } else if (sources.size() == 1) {
                        Path source = (Path)sources.stream().findFirst().orElseThrow();
                        Files.move(source, destination, new CopyOption[0]);
                        logger.debug("Moved {} to {}", (Object)source, (Object)destination);
                        return;
                    }
                    for (Path p : sources) {
                        InputStream inputStream = this.getInputStream(p);
                        try {
                            this.addAll(merged, this.bucketCodec.decode(inputStream), null);
                        }
                        finally {
                            if (inputStream == null) continue;
                            inputStream.close();
                        }
                    }
                    ByteArrayOutputStream outputStream = this.bucketCodec.encode(new ResiduePairIdentifierBucket(merged));
                    this.write(destination, outputStream);
                    outputStream.close();
                    for (Path source : sources) {
                        Files.delete(source);
                    }
                    logger.debug("Merged {} into {}", (Object)sources, (Object)destination);
                }
                catch (IOException e) {
                    throw new UncheckedIOException(e);
                }
            });
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    @Override
    public void clearTemporaryFiles() {
        try {
            logger.info("Collecting all temporary index files at {}", (Object)this.basePath);
            AtomicInteger counter = new AtomicInteger();
            try (Stream<Path> paths = this.temporaryFiles();){
                paths.peek(p -> this.progress(counter, 10000, "{} files deleted")).forEach(p -> {
                    try {
                        Files.delete(p);
                    }
                    catch (IOException e) {
                        throw new UncheckedIOException(e);
                    }
                });
            }
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private void write(Path path, ByteArrayOutputStream data) throws IOException {
        try (OutputStream outputStream = Files.newOutputStream(path, new OpenOption[0]);){
            OutputStream actual = this.gzipped ? new GZIPOutputStream(outputStream, 65536) : outputStream;
            data.writeTo(actual);
            actual.flush();
            actual.close();
        }
    }

    @Override
    public InvertedIndexBucket select(ResiduePairDescriptor residuePairDescriptor) {
        InvertedIndexBucket invertedIndexBucket;
        block8: {
            InputStream inputStream = this.getInputStream(residuePairDescriptor);
            try {
                invertedIndexBucket = this.bucketCodec.decode(inputStream);
                if (inputStream == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (inputStream != null) {
                        try {
                            inputStream.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    return InvertedIndexBucket.EMPTY_BUCKET;
                }
            }
            inputStream.close();
        }
        return invertedIndexBucket;
    }

    protected InputStream getInputStream(ResiduePairDescriptor residuePairDescriptor) throws IOException {
        return this.getInputStream(this.getPath(residuePairDescriptor));
    }

    private InputStream getInputStream(Path path) throws IOException {
        InputStream inputStream = Files.newInputStream(path, new OpenOption[0]);
        return this.gzipped ? new GZIPInputStream(inputStream, 65536) : new BufferedInputStream(inputStream, 65536);
    }

    private Path getPath(ResiduePairDescriptor residuePairDescriptor) {
        String bin = residuePairDescriptor.toString();
        String uberbin = bin.substring(0, 2);
        return this.basePath.resolve(uberbin).resolve(bin + this.extension);
    }

    private Path getPath(ResiduePairDescriptor residuePairDescriptor, int batchId) {
        Path actual = this.getPath(residuePairDescriptor);
        return actual.resolveSibling(actual.getFileName().toString() + "." + batchId);
    }

    private InvertedIndexBucket getBucket(ResiduePairDescriptor residuePairDescriptor) {
        InvertedIndexBucket invertedIndexBucket;
        block8: {
            InputStream inputStream = this.getInputStream(residuePairDescriptor);
            try {
                invertedIndexBucket = this.bucketCodec.decode(inputStream);
                if (inputStream == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (inputStream != null) {
                        try {
                            inputStream.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    return InvertedIndexBucket.EMPTY_BUCKET;
                }
            }
            inputStream.close();
        }
        return invertedIndexBucket;
    }

    @Override
    public void delete(Collection<Integer> removals) {
        try {
            logger.info("Removing {} structures from inverted index", (Object)removals.size());
            AtomicInteger counter = new AtomicInteger();
            this.indexFiles().peek(path -> this.progress(counter, 10000, "{} bins of inverted index cleaned")).map(this::createResiduePairDescriptor).forEach(residuePairDescriptor -> this.delete((ResiduePairDescriptor)residuePairDescriptor, removals));
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private void progress(AtomicInteger counter, int interval, String message) {
        int i = counter.incrementAndGet();
        if (i % interval == 0) {
            logger.info(message, (Object)i);
        }
    }

    private ResiduePairDescriptor createResiduePairDescriptor(Path path) {
        String name = path.toFile().getName();
        String[] split = name.split("\\.")[0].split("-");
        ResidueType residueType1 = OLC_LOOKUP.getOrDefault(split[0].substring(0, 1), null);
        ResidueType residueType2 = OLC_LOOKUP.getOrDefault(split[0].substring(1, 2), null);
        DistanceType d1 = DistanceType.ofIntRepresentation(Integer.parseInt(split[1]));
        DistanceType d2 = DistanceType.ofIntRepresentation(Integer.parseInt(split[2]));
        AngleType a = AngleType.ofIntRepresentation(Integer.parseInt(split[3]));
        return new ResiduePairDescriptor(residueType1, residueType2, d1, d2, a);
    }

    private void delete(ResiduePairDescriptor residuePairDescriptor, Collection<Integer> removals) {
        try {
            InvertedIndexBucket bucket = this.getBucket(residuePairDescriptor);
            Set<Integer> structureIndices = bucket.getStructureIndices();
            if (removals.stream().noneMatch(structureIndices::contains)) {
                return;
            }
            ResiduePairIdentifierBucket filteredBucket = this.removeByKey(bucket, removals);
            try (ByteArrayOutputStream outputStream = this.bucketCodec.encode(filteredBucket);){
                Path path = this.getPath(residuePairDescriptor);
                this.write(path, outputStream);
            }
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private void ensureDirectoriesExist() {
        try {
            List oneLetterCodes = Stream.of(ResidueType.values()).map(ResidueType::getInternalCode).sorted().collect(Collectors.toList());
            for (int i = 0; i < oneLetterCodes.size(); ++i) {
                for (int j = i; j < oneLetterCodes.size(); ++j) {
                    Path dir = this.basePath.resolve((String)oneLetterCodes.get(i) + (String)oneLetterCodes.get(j));
                    if (Files.exists(dir, new LinkOption[0])) continue;
                    Files.createDirectories(dir, new FileAttribute[0]);
                }
            }
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    @Override
    public Set<ResiduePairDescriptor> reportKnownDescriptors() {
        try {
            logger.info("Collecting all known descriptors at {}", (Object)this.basePath);
            AtomicInteger counter = new AtomicInteger();
            return this.indexFiles().peek(p -> this.progress(counter, 10000, "{} bins scanned")).map(this::createResiduePairDescriptor).collect(Collectors.toSet());
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    @Override
    public Set<Integer> reportKnownKeys() {
        try {
            logger.info("Collecting all known keys at {}", (Object)this.basePath);
            AtomicInteger counter = new AtomicInteger();
            return this.indexFiles().peek(p -> this.progress(counter, 10000, "{} bins scanned")).map(this::createResiduePairDescriptor).map(this::getBucket).map(Bucket::getStructureIndices).flatMap(Collection::stream).collect(Collectors.toSet());
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private Stream<Path> indexFiles() throws IOException {
        return this.files().filter(p -> p.getFileName().toString().endsWith(this.extension));
    }

    private Stream<Path> temporaryFiles() throws IOException {
        return this.files().filter(p -> this.tempExtension.matcher(p.getFileName().toString()).matches());
    }

    private Stream<Path> allFiles() throws IOException {
        return this.files().filter(p -> p.getFileName().toString().contains(this.extension));
    }

    private Stream<Path> files() throws IOException {
        if (!Files.exists(this.basePath, new LinkOption[0])) {
            return Stream.empty();
        }
        return ((Stream)Files.walk(this.basePath, FileVisitOption.FOLLOW_LINKS).parallel()).filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0]));
    }

    private void addAll(Map<Integer, Collection<ResiduePairIdentifier>> map, Bucket bucket, Collection<Integer> ignore) {
        while (bucket.hasNextStructure()) {
            bucket.moveStructure();
            int key = bucket.getStructureIndex();
            if (ignore != null && ignore.contains(key)) continue;
            if (map.containsKey(key)) {
                throw new IllegalStateException("Duplicate key: " + key);
            }
            Collection identifiers = map.computeIfAbsent(key, e -> new ArrayList());
            while (bucket.hasNextOccurrence()) {
                bucket.moveOccurrence();
                ResiduePairIdentifier residuePairIdentifier = bucket.getResiduePairIdentifier();
                identifiers.add(residuePairIdentifier);
            }
        }
    }

    private ResiduePairIdentifierBucket removeByKey(InvertedIndexBucket bucket, Collection<Integer> removals) {
        HashMap<Integer, Collection<ResiduePairIdentifier>> map = new HashMap<Integer, Collection<ResiduePairIdentifier>>();
        this.addAll(map, bucket, removals);
        return new ResiduePairIdentifierBucket(map);
    }
}

