/*
 * Decompiled with CFR 0.152.
 */
package org.rcsb.strucmotif.io;

import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.annotation.PostConstruct;
import org.rcsb.cif.schema.mm.MmCifFile;
import org.rcsb.strucmotif.config.InMemoryStrategy;
import org.rcsb.strucmotif.config.StrucmotifConfig;
import org.rcsb.strucmotif.domain.Pair;
import org.rcsb.strucmotif.domain.structure.Structure;
import org.rcsb.strucmotif.io.StructureDataProvider;
import org.rcsb.strucmotif.io.StructureReader;
import org.rcsb.strucmotif.io.StructureWriter;
import org.rcsb.strucmotif.math.Partition;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

@Service
public class StructureDataProviderImpl
implements StructureDataProvider {
    private static final Logger logger = LoggerFactory.getLogger(StructureDataProviderImpl.class);
    private final StructureReader structureReader;
    private final StructureWriter renumberedStructureWriter;
    private final StrucmotifConfig strucmotifConfig;
    private final String dataSource;
    private final Path renumberedDirectory;
    private final String extension;
    private boolean paths;
    private boolean caching;
    private Map<String, Structure> structureCache;

    @Autowired
    public StructureDataProviderImpl(StructureReader structureReader, StructureWriter structureWriter, StrucmotifConfig strucmotifConfig) {
        this.structureReader = structureReader;
        this.renumberedStructureWriter = structureWriter;
        this.strucmotifConfig = strucmotifConfig;
        this.dataSource = strucmotifConfig.getDataSource();
        this.renumberedDirectory = Paths.get(strucmotifConfig.getRootPath(), new String[0]).resolve("renumbered");
        this.extension = strucmotifConfig.isRenumberedGzip() ? ".bcif.gz" : ".bcif";
        logger.info("BinaryCIF data source is {} - CIF fetch URL: {} - precision: {} - gzipping: {}", new Object[]{strucmotifConfig.getDataSource(), strucmotifConfig.getCifFetchUrl(), strucmotifConfig.getRenumberedCoordinatePrecision(), strucmotifConfig.isRenumberedGzip()});
        this.paths = false;
        this.caching = false;
    }

    private void ensureRenumberedPathExists() {
        try {
            Files.createDirectories(this.renumberedDirectory, new FileAttribute[0]);
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private String prepareUri(String raw, String structureIdentifier) {
        String pdbId = structureIdentifier.toLowerCase();
        String pdbIdUc = pdbId.toUpperCase();
        String middle = pdbId.substring(1, 3);
        String middleUc = middle.toUpperCase();
        return raw.replace("{middle}", middle).replace("{MIDDLE}", middleUc).replace("{id}", pdbId).replace("{ID}", pdbIdUc);
    }

    private URL getCifFetchUrl(String structureIdentifier) {
        try {
            return new URL(this.prepareUri(this.strucmotifConfig.getCifFetchUrl(), structureIdentifier));
        }
        catch (MalformedURLException e) {
            throw new UncheckedIOException(e);
        }
    }

    private Path getOriginalStructurePath(String structureIdentifier) {
        return Paths.get(this.prepareUri(this.dataSource, structureIdentifier), new String[0]);
    }

    private Path getRenumberedStructurePath(String structureIdentifier) {
        return this.renumberedDirectory.resolve(structureIdentifier + this.extension);
    }

    private InputStream getRenumberedInputStream(String structureIdentifier) {
        try {
            return Files.newInputStream(this.getRenumberedStructurePath(structureIdentifier), new OpenOption[0]);
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    @Override
    @PostConstruct
    public void initializeRenumberedStructureCache() throws IOException {
        InMemoryStrategy strategy = this.strucmotifConfig.getInMemoryStrategy();
        if (strategy == InMemoryStrategy.OFF) {
            logger.info("Structure data will be read from file-system");
            return;
        }
        if (strategy == InMemoryStrategy.HEAP) {
            List files;
            logger.info("Structure data will be kept in memory - start loading...");
            this.caching = true;
            try (Stream<Path> pathStream = Files.walk(this.renumberedDirectory, new FileVisitOption[0]);){
                files = ((Stream)pathStream.parallel()).filter(path -> !Files.isDirectory(path, new LinkOption[0])).collect(Collectors.toList());
            }
            long start = System.nanoTime();
            this.structureCache = new HashMap<String, Structure>();
            int loadingChunkSize = this.strucmotifConfig.getLoadingChunkSize();
            Partition partitions = new Partition(files, loadingChunkSize);
            logger.info("Formed {} partitions of {} structures", (Object)partitions.size(), (Object)loadingChunkSize);
            for (int i = 0; i < partitions.size(); ++i) {
                String partitionContext = i + 1 + " / " + partitions.size();
                Object partition = partitions.get(i);
                logger.info("[{}] Start loading partition", (Object)partitionContext);
                Map<String, Structure> buffer = partition.parallelStream().map(this::loadRenumberedStructure).collect(Collectors.toMap(Pair::getFirst, Pair::getSecond));
                this.structureCache.putAll(buffer);
            }
            long time = (System.nanoTime() - start) / 1000L / 1000L / 1000L;
            long atoms = this.structureCache.values().stream().mapToLong(Structure::getAtomCount).sum();
            logger.info("Done caching structure data in {} seconds - {} atoms in {} structures held in memory", new Object[]{time, atoms, this.structureCache.size()});
        }
    }

    private Pair<String, Structure> loadRenumberedStructure(Path path) {
        try {
            String pdbId = path.toFile().getName().split("\\.")[0];
            Structure structure = this.readFromInputStream(Files.newInputStream(path, new OpenOption[0]));
            return new Pair<String, Structure>(pdbId, structure);
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    @Override
    public InputStream getOriginalInputStream(String structureIdentifier) {
        try {
            Path originalPath = this.getOriginalStructurePath(structureIdentifier);
            if (Files.isReadable(originalPath)) {
                return Files.newInputStream(originalPath, new OpenOption[0]);
            }
            return this.getCifFetchUrl(structureIdentifier).openStream();
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    @Override
    public Structure readFromInputStream(InputStream inputStream) {
        return this.structureReader.readFromInputStream(inputStream);
    }

    @Override
    public Structure readRenumbered(String structureIdentifier) {
        if (this.caching) {
            return this.structureCache.get(structureIdentifier);
        }
        return this.readFromInputStream(this.getRenumberedInputStream(structureIdentifier));
    }

    @Override
    public Structure readOriginal(String structureIdentifier) {
        return this.readFromInputStream(this.getOriginalInputStream(structureIdentifier));
    }

    @Override
    public Structure readSome(String structureIdentifier) {
        try {
            Path originalPath = this.getOriginalStructurePath(structureIdentifier);
            return this.readFromInputStream(Files.newInputStream(originalPath, new OpenOption[0]));
        }
        catch (IOException e1) {
            try {
                Path renumberedPath = this.getRenumberedStructurePath(structureIdentifier);
                if (Files.isReadable(renumberedPath)) {
                    return this.readFromInputStream(Files.newInputStream(renumberedPath, new OpenOption[0]));
                }
                return this.readFromInputStream(this.getCifFetchUrl(structureIdentifier).openStream());
            }
            catch (IOException e2) {
                throw new UncheckedIOException(e2);
            }
        }
    }

    @Override
    public void writeRenumbered(String structureIdentifier, MmCifFile mmCifFile) {
        if (!this.paths) {
            this.ensureRenumberedPathExists();
            this.paths = true;
        }
        this.renumberedStructureWriter.write(mmCifFile, this.getRenumberedStructurePath(structureIdentifier));
    }

    @Override
    public void deleteRenumbered(String structureIdentifier) {
        try {
            Path renumberedPath = this.getRenumberedStructurePath(structureIdentifier);
            if (Files.exists(renumberedPath, new LinkOption[0])) {
                Files.delete(this.getRenumberedStructurePath(structureIdentifier));
            }
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }
}

