package reactivemongo.api.collections

import scala.language.higherKinds

import scala.concurrent.duration.FiniteDuration

import reactivemongo.api.{
  Collation,
  Cursor,
  CursorOptions,
  CursorProducer,
  ReadConcern,
  ReadPreference,
  SerializationPack
}

import reactivemongo.api.commands.{
  CollectionCommand,
  CommandCodecs,
  CommandWithPack,
  CommandWithResult,
  ResolvedCollectionCommand,
  WriteConcern
}
import reactivemongo.core.protocol.MongoWireVersion

private[collections] trait Aggregator[P <: SerializationPack with Singleton] {
  collection: GenericCollection[P] with HintFactory[P] =>

  // ---

  /**
   * @see [[reactivemongo.api.commands.AggregationFramework.PipelineOperator]]
   */
  final class AggregatorContext[T](
    val firstOperator: PipelineOperator,
    val otherOperators: List[PipelineOperator],
    val explain: Boolean,
    val allowDiskUse: Boolean,
    val bypassDocumentValidation: Boolean,
    val readConcern: ReadConcern,
    val writeConcern: WriteConcern,
    val readPreference: ReadPreference,
    val batchSize: Option[Int],
    val cursorOptions: CursorOptions,
    val maxTime: Option[FiniteDuration],
    val reader: pack.Reader[T],
    val hint: Option[Hint[pack.type]],
    val comment: Option[String],
    val collation: Option[Collation]) {

    @deprecated("Use `maxTime`", "0.19.8")
    val maxTimeMS: Option[Long] = maxTime.map(_.toMillis)

    def prepared[AC[_] <: Cursor.WithOps[_]](
      implicit
      cp: CursorProducer.Aux[T, AC]): Aggregator[T, AC] =
      new Aggregator[T, AC](this, cp)
  }

  final class Aggregator[T, AC[_] <: Cursor[_]](
    val context: AggregatorContext[T],
    val cp: CursorProducer.Aux[T, AC]) {

    private def ver = db.connectionState.metadata.maxWireVersion

    final def cursor: AC[T] = {
      def batchSz = context.batchSize.getOrElse(defaultCursorBatchSize)
      implicit def writer = commandWriter[T]
      implicit def aggReader: pack.Reader[T] = context.reader

      val cmd = new Aggregate[T](
        context.firstOperator, context.otherOperators, context.explain,
        context.allowDiskUse, batchSz, ver,
        context.bypassDocumentValidation,
        context.readConcern, context.writeConcern,
        context.hint, context.comment, context.collation)

      val cursor = runner.cursor[T, Aggregate[T]](
        collection, cmd, context.cursorOptions,
        context.readPreference, context.maxTimeMS)

      cp.produce(cursor)
    }
  }

  /**
   * @param pipeline the sequence of MongoDB aggregation operations
   * @param explain specifies to return the information on the processing of the pipeline
   * @param allowDiskUse enables writing to temporary files
   * @param batchSize the batch size
   * @param bypassDocumentValidation available only if you specify the \$out aggregation operator
   * @param readConcern the read concern (since MongoDB 3.2)
   */
  private final class Aggregate[T](
    val operator: PipelineOperator,
    val pipeline: Seq[PipelineOperator],
    val explain: Boolean = false,
    val allowDiskUse: Boolean,
    val batchSize: Int,
    val wireVersion: MongoWireVersion,
    val bypassDocumentValidation: Boolean,
    val readConcern: ReadConcern,
    val writeConcern: WriteConcern,
    val hint: Option[Hint[pack.type]],
    val comment: Option[String],
    val collation: Option[Collation]) extends CollectionCommand
    with CommandWithPack[pack.type]
    with CommandWithResult[T]

  private type AggregateCmd[T] = ResolvedCollectionCommand[Aggregate[T]]

  private def commandWriter[T]: pack.Writer[AggregateCmd[T]] = {
    val builder = pack.newBuilder
    val session = collection.db.session.filter( // TODO#1.1: Remove
      _ => (version.compareTo(MongoWireVersion.V36) >= 0))

    val writeWriteConcern = CommandCodecs.writeWriteConcern(builder)
    val writeCollation = Collation.serializeWith(pack, _: Collation)(builder)

    pack.writer[AggregateCmd[T]] { agg =>
      import builder.{ boolean, document, elementProducer => element }
      import agg.{ command => cmd }

      val pipeline = builder.array(
        cmd.operator.makePipe,
        cmd.pipeline.map(_.makePipe))

      lazy val isOut: Boolean = cmd.pipeline.lastOption.exists {
        case BatchCommands.AggregationFramework.Out(_) => true
        case _                                         => false
      }

      val writeReadConcern =
        CommandCodecs.writeSessionReadConcern(builder)(session)

      val elements = Seq.newBuilder[pack.ElementProducer]

      elements ++= Seq(
        element("aggregate", builder.string(agg.collection)),
        element("pipeline", pipeline),
        element("explain", boolean(cmd.explain)),
        element("allowDiskUse", boolean(cmd.allowDiskUse)),
        element("cursor", document(Seq(
          element("batchSize", builder.int(cmd.batchSize))))))

      if (cmd.wireVersion >= MongoWireVersion.V32) {
        elements += element("bypassDocumentValidation", boolean(
          cmd.bypassDocumentValidation))

        elements ++= writeReadConcern(cmd.readConcern)
      }

      cmd.comment.foreach { comment =>
        elements += element("comment", builder.string(comment))
      }

      cmd.hint.foreach {
        case HintString(str) =>
          elements += element("hint", builder.string(str))

        case HintDocument(doc) =>
          elements += element("hint", doc)
      }

      cmd.collation.foreach { collation =>
        elements += element("collation", writeCollation(collation))
      }

      if (cmd.wireVersion >= MongoWireVersion.V36 && isOut) {
        elements += element("writeConcern", writeWriteConcern(cmd.writeConcern))
      }

      document(elements.result())
    }
  }
}
