/*
 * Copyright 2012-2013 Stephane Godbillon (@sgodbillon) and Zenexity
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package reactivemongo.api.indexes

import reactivemongo.core.protocol.MongoWireVersion

import reactivemongo.api.{
  DB,
  DBMetaCommands,
  Collation,
  Cursor,
  CursorProducer,
  ReadPreference,
  Serialization,
  SerializationPack
}

import reactivemongo.api.commands.{ CommandError, DropIndexes, WriteResult }

import scala.concurrent.{ Future, ExecutionContext }

/**
 * Indexes manager at database level.
 *
 * @define createDescription Creates the given index
 * @define dropDescription Drops the specified index
 * @define collectionNameParam the collection name
 * @define nsIndexToCreate the index to create
 * @define droppedCount The number of indexes that were dropped.
 */
sealed trait IndexesManager {

  /**
   * Lists all the index on this database.
   *
   * {{{
   * import scala.concurrent.{ ExecutionContext, Future }
   *
   * import reactivemongo.api.DefaultDB
   * import reactivemongo.api.indexes.NSIndex
   *
   * def listIndexes(db: DefaultDB)(
   *   implicit ec: ExecutionContext): Future[List[String]] =
   *   db.indexesManager.list().map(_.flatMap { ni: NSIndex =>
   *     ni.index.name.toList
   *   })
   * }}}
   */
  def list(): Future[List[NSIndex]]

  /**
   * $createDescription only if it does not exist on this database.
   *
   * The following rules are used to check the matching index:
   * - if `nsIndex.isDefined`, it checks using the index name,
   * - otherwise it checks using the key.
   *
   * {{{
   * import scala.concurrent.{ ExecutionContext, Future }
   *
   * import reactivemongo.api.DefaultDB
   * import reactivemongo.api.indexes.NSIndex
   *
   * def ensureIndexes(
   *   db: DefaultDB, is: Seq[NSIndex])(
   *   implicit ec: ExecutionContext): Future[Unit] =
   *   Future.sequence(
   *     is.map(idx => db.indexesManager.ensure(idx))).map(_ => {})
   * }}}
   *
   * _Warning_: given the options you choose, and the data to index,
   * it can be a long and blocking operation on the database.
   * You should really consider reading [[http://www.mongodb.org/display/DOCS/Indexes]] before doing this, especially in production.
   *
   * @param nsIndex $nsIndexToCreate
   * @return true if the index was created, false if it already exists.
   */
  def ensure(nsIndex: NSIndex): Future[Boolean]

  /**
   * $createDescription.
   *
   * {{{
   * import scala.concurrent.{ ExecutionContext, Future }
   *
   * import reactivemongo.api.DefaultDB
   * import reactivemongo.api.indexes.NSIndex
   *
   * def createIndexes(
   *   db: DefaultDB, is: Seq[NSIndex])(
   *   implicit ec: ExecutionContext): Future[Unit] =
   *   Future.sequence(
   *     is.map(idx => db.indexesManager.create(idx))).map(_ => {})
   * }}}
   *
   * _Warning_: given the options you choose, and the data to index,
   * it can be a long and blocking operation on the database.
   * You should really consider reading [[http://www.mongodb.org/display/DOCS/Indexes]] before doing this, especially in production.
   *
   * @param nsIndex $nsIndexToCreate
   */
  def create(nsIndex: NSIndex): Future[WriteResult]

  /**
   * $dropDescription.
   *
   * {{{
   * import scala.concurrent.{ ExecutionContext, Future }
   *
   * import reactivemongo.api.DefaultDB
   * import reactivemongo.api.indexes.NSIndex
   *
   * def dropIndex(db: DefaultDB, idx: NSIndex)(
   *   implicit ec: ExecutionContext): Future[Int] =
   *   db.indexesManager.drop(idx)
   * }}}
   *
   * @return $droppedCount
   */
  def drop(nsIndex: NSIndex): Future[Int] =
    drop(nsIndex.collectionName, nsIndex.index.eventualName)

  /**
   * $dropDescription on the given collection.
   *
   * {{{
   * import scala.concurrent.{ ExecutionContext, Future }
   * import reactivemongo.api.DefaultDB
   *
   * def dropIndex(db: DefaultDB, name: String)(
   *   implicit ec: ExecutionContext): Future[Int] =
   *   db.indexesManager.drop("myColl", name)
   * }}}
   *
   * @param collectionName $collectionNameParam
   * @param indexName the name of the index to be dropped
   * @return $droppedCount
   */
  def drop(collectionName: String, indexName: String): Future[Int]

  /**
   * Drops all the indexes on the specified collection.
   *
   * {{{
   * import scala.concurrent.{ ExecutionContext, Future }
   * import reactivemongo.api.DefaultDB
   *
   * def dropAllIndexes(db: DefaultDB)(
   *   implicit ec: ExecutionContext): Future[Int] =
   *   db.indexesManager.dropAll("myColl")
   * }}}
   *
   * @param collectionName $collectionNameParam
   * @return $droppedCount
   */
  def dropAll(collectionName: String): Future[Int]

  /**
   * Returns a manager for the specified collection.
   *
   * {{{
   * import scala.concurrent.{ ExecutionContext, Future }
   * import reactivemongo.api.DefaultDB
   *
   * def countCollIndexes(db: DefaultDB, collName: String)(
   *   implicit ec: ExecutionContext): Future[Int] =
   *   db.indexesManager.onCollection(collName).list().map(_.size)
   * }}}
   *
   * @param collectionName $collectionNameParam
   */
  def onCollection(@deprecatedName(Symbol("name")) collectionName: String): CollectionIndexesManager
}

/**
 * A helper class to manage the indexes on a Mongo 2.x database.
 *
 * @param db The subject database.
 */
final class LegacyIndexesManager(db: DB)(
  implicit
  ec: ExecutionContext) extends IndexesManager {

  val collection = db("system.indexes")(Serialization.defaultCollectionProducer)
  import collection.pack

  private lazy val builder = pack.newBuilder

  def list(): Future[List[NSIndex]] = collection.find(builder.document(Seq.empty), Option.empty[pack.Document]).cursor(db.connection.options.readPreference)(IndexesManager.nsIndexReader, CursorProducer.defaultCursorProducer).collect[List](-1, Cursor.FailOnError[List[NSIndex]]())

  def ensure(nsIndex: NSIndex): Future[Boolean] = {
    import builder.string

    val query = builder.document(Seq(
      builder.elementProducer("ns", string(nsIndex.namespace)),
      builder.elementProducer("name", string(nsIndex.index.eventualName))))

    collection.find(query, Option.empty[pack.Document]).one.flatMap { idx =>
      if (!idx.isDefined) {
        create(nsIndex).map(_ => true)
      } else {
        Future.successful(false)
      }
    }
  }

  private[reactivemongo] implicit lazy val nsIndexWriter =
    IndexesManager.nsIndexWriter(pack)

  def create(nsIndex: NSIndex): Future[WriteResult] =
    collection.insert.one(nsIndex)

  private implicit lazy val dropWriter = DropIndexes.writer(pack)

  private lazy val dropReader = DropIndexes.reader(pack)

  def drop(collectionName: String, indexName: String): Future[Int] = {
    implicit def reader = dropReader

    db.collection(collectionName).
      runValueCommand(DropIndexes(indexName), ReadPreference.primary)
  }

  def dropAll(collectionName: String): Future[Int] = drop(collectionName, "*")

  def onCollection(@deprecatedName(Symbol("name")) collectionName: String): CollectionIndexesManager = new LegacyCollectionIndexesManager(db.name, collectionName, this)
}

/**
 * A helper class to manage the indexes on a Mongo 3.x database.
 *
 * @param db the subject database
 */
final class DefaultIndexesManager(db: DB with DBMetaCommands)(
  implicit
  ec: ExecutionContext) extends IndexesManager {

  private def listIndexes(collections: List[String], indexes: List[NSIndex]): Future[List[NSIndex]] = collections match {
    case c :: cs => onCollection(c).list().flatMap(ix =>
      listIndexes(cs, indexes ++ ix.map(NSIndex(s"${db.name}.$c", _))))

    case _ => Future.successful(indexes)
  }

  def list(): Future[List[NSIndex]] =
    db.collectionNames.flatMap(listIndexes(_, Nil))

  def ensure(nsIndex: NSIndex): Future[Boolean] =
    onCollection(nsIndex.collectionName).ensure(nsIndex.index)

  def create(nsIndex: NSIndex): Future[WriteResult] =
    onCollection(nsIndex.collectionName).create(nsIndex.index)

  def drop(collectionName: String, indexName: String): Future[Int] =
    onCollection(collectionName).drop(indexName)

  def dropAll(collectionName: String): Future[Int] =
    onCollection(collectionName).dropAll()

  def onCollection(@deprecatedName(Symbol("name")) collectionName: String): CollectionIndexesManager = new DefaultCollectionIndexesManager(db, collectionName)
}

/**
 * @define the index to create
 * @define droppedCount The number of indexes that were dropped.
 * @define indexToCreate the index to create
 * @define createDescription Creates the given index
 */
sealed trait CollectionIndexesManager {
  /**
   * Lists the indexes for the current collection.
   *
   * {{{
   * import scala.concurrent.{ ExecutionContext, Future }
   * import reactivemongo.api.CollectionMetaCommands
   *
   * def listIndexes(coll: CollectionMetaCommands)(
   *   implicit ec: ExecutionContext): Future[List[String]] =
   *   coll.indexesManager.list().map(_.flatMap { idx =>
   *     idx.name.toList
   *   })
   * }}}
   */
  def list(): Future[List[Index]]

  /**
   * $createDescription only if it does not exist on this collection.
   *
   * The following rules are used to check the matching index:
   * - if `nsIndex.isDefined`, it checks using the index name,
   * - otherwise it checks using the key.
   *
   * {{{
   * import scala.concurrent.{ ExecutionContext, Future }
   *
   * import reactivemongo.api.CollectionMetaCommands
   * import reactivemongo.api.indexes.Index
   *
   * def ensureIndexes(
   *   coll: CollectionMetaCommands, is: Seq[Index])(
   *   implicit ec: ExecutionContext): Future[Unit] =
   *   Future.sequence(
   *     is.map(idx => coll.indexesManager.ensure(idx))).map(_ => {})
   * }}}
   *
   * _Warning_: given the options you choose, and the data to index,
   * it can be a long and blocking operation on the database.
   * You should really consider reading [[http://www.mongodb.org/display/DOCS/Indexes]] before doing this, especially in production.
   *
   * @param index $indexToCreate
   *
   * @return true if the index was created, false if it already exists.
   */
  def ensure(index: Index): Future[Boolean]

  /**
   * $createDescription.
   *
   * {{{
   * import scala.concurrent.{ ExecutionContext, Future }
   *
   * import reactivemongo.api.CollectionMetaCommands
   * import reactivemongo.api.indexes.Index
   *
   * def createIndexes(
   *   coll: CollectionMetaCommands, is: Seq[Index])(
   *   implicit ec: ExecutionContext): Future[Unit] =
   *   Future.sequence(
   *     is.map(idx => coll.indexesManager.create(idx))).map(_ => {})
   * }}}
   *
   * _Warning_: given the options you choose, and the data to index,
   * it can be a long and blocking operation on the database.
   * You should really consider reading [[http://www.mongodb.org/display/DOCS/Indexes]] before doing this, especially in production.
   *
   * @param index $indexToCreate
   */
  def create(index: Index): Future[WriteResult]

  /**
   * Drops the given index on that collection.
   *
   * {{{
   * import scala.concurrent.{ ExecutionContext, Future }
   *
   * import reactivemongo.api.CollectionMetaCommands
   *
   * def dropIndex(coll: CollectionMetaCommands, name: String)(
   *   implicit ec: ExecutionContext): Future[Int] =
   *   coll.indexesManager.drop(name)
   * }}}
   *
   * @param indexName the name of the index to be dropped
   * @return $droppedCount
   */
  def drop(indexName: String): Future[Int]

  /**
   * Drops all the indexes on that collection.
   *
   * {{{
   * import scala.concurrent.{ ExecutionContext, Future }
   * import reactivemongo.api.CollectionMetaCommands
   *
   * def dropAllIndexes(coll: CollectionMetaCommands)(
   *   implicit ec: ExecutionContext): Future[Int] =
   *   coll.indexesManager.dropAll()
   * }}}
   *
   * @return $droppedCount
   */
  def dropAll(): Future[Int]
}

private class LegacyCollectionIndexesManager(
  db: String, collectionName: String, legacy: LegacyIndexesManager)(
  implicit
  ec: ExecutionContext) extends CollectionIndexesManager {

  val fqName = db + "." + collectionName

  def list(): Future[List[Index]] =
    legacy.list.map(_.filter(_.namespace == fqName).map(_.index))

  def ensure(index: Index): Future[Boolean] =
    legacy.ensure(NSIndex(fqName, index))

  def create(index: Index): Future[WriteResult] =
    legacy.create(NSIndex(fqName, index))

  def drop(indexName: String): Future[Int] =
    legacy.drop(collectionName, indexName)

  def dropAll(): Future[Int] = legacy.dropAll(collectionName)
}

private class DefaultCollectionIndexesManager(db: DB, collectionName: String)(
  implicit
  ec: ExecutionContext) extends CollectionIndexesManager {

  import reactivemongo.api.commands.{
    CreateIndexes,
    Command,
    ListIndexes
  }

  import Serialization.{ internalSerializationPack, writeResultReader }

  private lazy val collection = db(collectionName)
  private lazy val listCommand = ListIndexes(db.name)

  private lazy val runner =
    Command.run(internalSerializationPack, db.failoverStrategy)

  private implicit lazy val listWriter =
    ListIndexes.writer(internalSerializationPack)

  private implicit lazy val indexReader =
    IndexesManager.indexReader[Serialization.Pack](
      internalSerializationPack)

  private implicit lazy val listReader =
    ListIndexes.reader(internalSerializationPack)

  def list(): Future[List[Index]] =
    runner(collection, listCommand, ReadPreference.primary).recoverWith {
      case CommandError.Code(26 /* no database or collection */ ) =>
        Future.successful(List.empty[Index])

      case err => Future.failed(err)
    }

  def ensure(index: Index): Future[Boolean] = list().flatMap { indexes =>
    val idx = index.name match {
      case Some(n) => indexes.find(_.name.exists(_ == n))
      case _       => indexes.find(_.key == index.key)
    }

    if (!idx.isDefined) {
      create(index).map(_ => true)
    } else {
      Future.successful(false)
    }
  }

  private implicit val createWriter =
    CreateIndexes.writer(internalSerializationPack)

  def create(index: Index): Future[WriteResult] =
    runner(
      collection,
      CreateIndexes(db.name, List(index)),
      ReadPreference.primary)

  private implicit def dropWriter = IndexesManager.dropWriter
  private implicit def dropReader = IndexesManager.dropReader

  def drop(indexName: String): Future[Int] = {
    runner(
      collection, DropIndexes(indexName), ReadPreference.primary).map(_.value)
  }

  @inline def dropAll(): Future[Int] = drop("*")
}

/** Factory for indexes manager scoped with a specified collection. */
object CollectionIndexesManager {
  /**
   * Returns an indexes manager for specified collection.
   *
   * @param db the database
   * @param collectionName the collection name
   */
  def apply(db: DB, collectionName: String)(implicit ec: ExecutionContext): CollectionIndexesManager = {
    val wireVer = db.connectionState.metadata.maxWireVersion

    if (wireVer >= MongoWireVersion.V30) {
      new DefaultCollectionIndexesManager(db, collectionName)
    } else new LegacyCollectionIndexesManager(db.name, collectionName,
      new LegacyIndexesManager(db))
  }
}

object IndexesManager {
  /**
   * Returns an indexes manager for specified database.
   *
   * @param db the database
   */
  def apply(db: DB with DBMetaCommands)(implicit ec: ExecutionContext): IndexesManager = {
    val wireVer = db.connectionState.metadata.maxWireVersion

    if (wireVer >= MongoWireVersion.V30) new DefaultIndexesManager(db)
    else new LegacyIndexesManager(db)
  }

  @deprecated("Internal: will be made private", "0.19.0")
  object NSIndexWriter extends reactivemongo.bson.BSONDocumentWriter[NSIndex] {
    private val underlying =
      nsIndexWriter(reactivemongo.api.BSONSerializationPack)

    def write(nsIndex: NSIndex): reactivemongo.bson.BSONDocument =
      underlying.write(nsIndex)
  }

  private[reactivemongo] def nsIndexWriter[P <: SerializationPack](pack: P): pack.Writer[NSIndex] = {
    val builder = pack.newBuilder
    val decoder = pack.newDecoder
    val writeIndexType = IndexType.write(pack)(builder)
    val writeCollation = Collation.serializeWith(pack, _: Collation)(builder)

    import builder.{ boolean, document, elementProducer => element, string }

    pack.writer[NSIndex] { nsIndex =>
      import nsIndex.index

      if (index.key.isEmpty) {
        throw new RuntimeException("the key should not be empty!")
      }

      val elements = Seq.newBuilder[pack.ElementProducer]

      elements ++= Seq(
        element("ns", string(nsIndex.namespace)),
        element("name", string(index.eventualName)),
        element("key", document(index.key.collect {
          case (k, v) => element(k, writeIndexType(v))
        })))

      if (index.background) {
        elements += element("background", boolean(true))
      }

      if (index.sparse) {
        elements += element("sparse", boolean(true))
      }

      index.expireAfterSeconds.foreach { sec =>
        elements += element("expireAfterSeconds", builder.int(sec))
      }

      index.storageEngine.map(index.pack.bsonValue).foreach {
        case pack.IsDocument(conf) =>
          elements += element("storageEngine", conf)

        case _ => ()
      }

      index.weights.map(index.pack.bsonValue).foreach {
        case pack.IsDocument(w) =>
          elements += element("weights", w)

        case _ => ()
      }

      index.defaultLanguage.foreach { lang =>
        elements += element("default_language", builder.string(lang))
      }

      index.languageOverride.foreach { lang =>
        elements += element("language_override", builder.string(lang))
      }

      index.textIndexVersion.foreach { ver =>
        elements += element("textIndexVersion", builder.int(ver))
      }

      index._2dsphereIndexVersion.foreach { ver =>
        elements += element("2dsphereIndexVersion", builder.int(ver))
      }

      index.bits.foreach { bits =>
        elements += element("bits", builder.int(bits))
      }

      index.min.foreach { min =>
        elements += element("min", builder.double(min))
      }

      index.max.foreach { max =>
        elements += element("max", builder.double(max))
      }

      index.bucketSize.foreach { size =>
        elements += element("bucketSize", builder.double(size))
      }

      index.collation.foreach { collation =>
        elements += element("collation", writeCollation(collation))
      }

      index.wildcardProjection.map(index.pack.bsonValue).foreach {
        case pack.IsDocument(projection) =>
          elements += element("wildcardProjection", projection)

        case _ => ()
      }

      if (index.unique) {
        elements += element("unique", boolean(true))
      }

      index.partialFilter.foreach { partialFilter =>
        elements += element(
          "partialFilterExpression", pack.document(partialFilter))
      }

      val opts = pack.document(index.options)
      decoder.names(opts).foreach { nme =>
        decoder.get(opts, nme).foreach { v =>
          elements += element(nme, v)
        }
      }

      document(elements.result())
    }
  }

  @deprecated("Internal: will be made private", "0.19.0")
  object IndexReader extends reactivemongo.bson.BSONDocumentReader[Index] {
    private val underlying =
      indexReader(reactivemongo.api.BSONSerializationPack)

    def read(doc: reactivemongo.bson.BSONDocument): Index = underlying.read(doc)
  }

  private[reactivemongo] def indexReader[P <: SerializationPack](pack: P): pack.Reader[Index] = {
    val decoder = pack.newDecoder
    val builder = pack.newBuilder
    val readCollation = Collation.read(pack)

    import decoder.{ booleanLike, child, double, int, string }

    pack.reader[Index] { doc =>
      child(doc, "key").fold[Index](
        throw new Exception("the key must be defined")) { k =>
          val ks = decoder.names(k).flatMap { nme =>
            decoder.get(k, nme).map { v =>
              nme -> IndexType(pack.bsonValue(v))
            }
          }

          val key = child(doc, "weights").fold(ks) { w =>
            val fields = decoder.names(w)

            (ks, fields).zipped.map {
              case ((_, tpe), name) => name -> tpe
            }
          }.toSeq

          val name = string(doc, "name")
          val unique = booleanLike(doc, "unique").getOrElse(false)
          val background = booleanLike(doc, "background").getOrElse(false)
          val dropDups = booleanLike(doc, "dropDups").getOrElse(false)
          val sparse = booleanLike(doc, "sparse").getOrElse(false)
          val expireAfterSeconds = int(doc, "expireAfterSeconds")
          val storageEngine = child(doc, "storageEngine")
          val weights = child(doc, "weights")
          val defaultLanguage = string(doc, "default_language")
          val languageOverride = string(doc, "language_override")
          val textIndexVersion = int(doc, "textIndexVersion")
          val sphereIndexVersion = int(doc, "2dsphereIndexVersion")
          val bits = int(doc, "bits")
          val min = double(doc, "min")
          val max = double(doc, "max")
          val bucketSize = double(doc, "bucketSize")
          val wildcardProjection = child(doc, "wildcardProjection")
          val collation = child(doc, "collation").flatMap(readCollation)
          val version = int(doc, "v")

          val options = builder.document(decoder.names(doc).flatMap {
            case "ns" | "key" | "name" | "unique" | "background" |
              "dropDups" | "sparse" | "v" | "partialFilterExpression" |
              "expireAfterSeconds" | "storageEngine" | "weights" |
              "defaultLanguage" | "languageOverride" | "textIndexVersion" |
              "2dsphereIndexVersion" | "bits" | "min" | "max" |
              "bucketSize" | "collation" | "wildcardProjection" =>
              Seq.empty[pack.ElementProducer]

            case nme =>
              decoder.get(doc, nme).map { v =>
                builder.elementProducer(nme, v)
              }
          }.toSeq)

          val partialFilter =
            child(doc, "partialFilterExpression")

          Index(pack)(key, name, unique, background, dropDups,
            sparse, expireAfterSeconds, storageEngine, weights,
            defaultLanguage, languageOverride, textIndexVersion,
            sphereIndexVersion, bits, min, max, bucketSize, collation,
            wildcardProjection, version, partialFilter, options)
        }
    }
  }

  @deprecated("Internal: will be made private", "0.19.0")
  object NSIndexReader extends reactivemongo.bson.BSONDocumentReader[NSIndex] {
    private val underlying =
      nsIndexReader(reactivemongo.api.BSONSerializationPack)

    def read(doc: reactivemongo.bson.BSONDocument): NSIndex =
      underlying.read(doc)
  }

  private[reactivemongo] def nsIndexReader[P <: SerializationPack](pack: P): pack.Reader[NSIndex] = {
    val decoder = pack.newDecoder
    val indexReader: pack.Reader[Index] = this.indexReader(pack)

    pack.reader[NSIndex] { doc =>
      decoder.string(doc, "ns").fold[NSIndex](
        throw new Exception("the namespace ns must be defined")) { ns =>
          NSIndex(ns, pack.deserialize(doc, indexReader))
        }
    }
  }

  private[api] implicit lazy val nsIndexReader =
    nsIndexReader[Serialization.Pack](Serialization.internalSerializationPack)

  private[api] lazy val dropWriter =
    DropIndexes.writer(Serialization.internalSerializationPack)

  private[api] lazy val dropReader =
    DropIndexes.reader(Serialization.internalSerializationPack)

}
