/**
 * Copyright 2016-2018 The Reaktivity Project
 *
 * The Reaktivity Project licenses this file to you under the Apache License,
 * version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at:
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */
package org.reaktivity.command.log.internal;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.stream.Stream;

import org.agrona.LangUtil;
import org.reaktivity.command.log.internal.layouts.StreamsLayout;
import org.reaktivity.command.log.internal.spy.RingBufferSpy;
import org.reaktivity.reaktor.internal.ReaktorConfiguration;

public final class LogQueueDepthCommand implements Runnable
{
    private final Path directory;
    private final boolean verbose;
    private final boolean separator;
    private final Logger out;

    private final Map<Path, StreamsLayout> layoutsByPath;

    public LogQueueDepthCommand(
        ReaktorConfiguration config,
        Logger out,
        boolean verbose,
        boolean separator)
    {
        this.directory = config.directory();
        this.out = out;
        this.verbose = verbose;
        this.separator = separator;
        this.layoutsByPath = new LinkedHashMap<>();
    }

    private boolean isStreamsFile(
        Path path)
    {
        return path.getNameCount() - directory.getNameCount() == 2 &&
                "streams".equals(path.getName(path.getNameCount() - 1).toString()) &&
                Files.isRegularFile(path);
    }

    private void onDiscovered(
        Path path)
    {
        if (verbose)
        {
            out.printf("Discovered: %s\n", path);
        }
    }

    private void displayQueueDepth(
        Path path)
    {

        StreamsLayout layout = layoutsByPath.computeIfAbsent(path, this::newStreamsLayout);
        String nukleus = path.getName(path.getNameCount() - 2).toString();
        displayQueueDepth(nukleus, "streams", layout.streamsBuffer());
    }

    private StreamsLayout newStreamsLayout(
        Path path)
    {
        return new StreamsLayout.Builder()
                .path(path)
                .readonly(true)
                .build();
    }

    private void displayQueueDepth(
        String nukleus,
        String type,
        RingBufferSpy buffer)
    {
        // read consumer position first for pessimistic queue depth
        long consumerAt = buffer.consumerPosition();
        long producerAt = buffer.producerPosition();

        final String valueFormat = separator ? ",d" : "d";

        out.printf("{\"nukleus\":\"%s\", " +
                    "\"head\":%" + valueFormat + ", " +
                    "\"tail\":%" + valueFormat + ", " +
                    "\"depth\":%" + valueFormat + "}\n",
                    nukleus, consumerAt, producerAt, producerAt - consumerAt);
    }

    @Override
    public void run()
    {
        try (Stream<Path> files = Files.walk(directory, 3))
        {
            files.filter(this::isStreamsFile)
                 .peek(this::onDiscovered)
                 .forEach(this::displayQueueDepth);
            out.printf("\n");
        }
        catch (IOException ex)
        {
            LangUtil.rethrowUnchecked(ex);
        }
    }
}
