/**
 * Copyright 2016-2018 The Reaktivity Project
 *
 * The Reaktivity Project licenses this file to you under the Apache License,
 * version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at:
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */
package org.reaktivity.command.log.internal;

import static java.util.concurrent.TimeUnit.MILLISECONDS;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.stream.Stream;

import org.agrona.LangUtil;
import org.agrona.collections.Long2LongHashMap;
import org.agrona.concurrent.BackoffIdleStrategy;
import org.agrona.concurrent.IdleStrategy;
import org.reaktivity.command.log.internal.labels.LabelManager;
import org.reaktivity.command.log.internal.layouts.StreamsLayout;
import org.reaktivity.reaktor.internal.ReaktorConfiguration;

public final class LogStreamsCommand implements Runnable
{
    private static final long MAX_PARK_NS = MILLISECONDS.toNanos(100L);
    private static final long MIN_PARK_NS = MILLISECONDS.toNanos(1L);
    private static final int MAX_YIELDS = 30;
    private static final int MAX_SPINS = 20;

    private final Path directory;
    private final LabelManager labels;
    private final boolean verbose;
    private final boolean continuous;
    private final Logger out;
    private final Long2LongHashMap budgets;
    private final Long2LongHashMap timestamps;

    private long nextTimestamp = Long.MAX_VALUE;

    LogStreamsCommand(
        ReaktorConfiguration config,
        Logger out,
        boolean verbose,
        boolean continuous)
    {
        this.directory = config.directory();
        this.labels = new LabelManager(directory);
        this.verbose = verbose;
        this.continuous = continuous;
        this.out = out;
        this.budgets = new Long2LongHashMap(-1L);
        this.timestamps = new Long2LongHashMap(-1L);
    }

    private boolean isStreamsFile(
        Path path)
    {
        return path.getNameCount() - directory.getNameCount() == 2 &&
               "streams".equals(path.getName(path.getNameCount() - 1).toString()) &&
               Files.isRegularFile(path);
    }

    private LoggableStream newLoggable(
        Path path)
    {
        StreamsLayout layout = new StreamsLayout.Builder()
                .path(path)
                .readonly(true)
                .build();

        return new LoggableStream(labels, budgets, layout, out, verbose, timestamps, this::nextTimestamp);
    }

    private void onDiscovered(
        Path path)
    {
        if (verbose)
        {
            out.printf("Discovered: %s\n", path);
        }
    }

    @Override
    public void run()
    {
        try (Stream<Path> files = Files.walk(directory, 3))
        {
            LoggableStream[] loggables = files.filter(this::isStreamsFile)
                 .peek(this::onDiscovered)
                 .map(this::newLoggable)
                 .toArray(LoggableStream[]::new);

            final IdleStrategy idleStrategy = new BackoffIdleStrategy(MAX_SPINS, MAX_YIELDS, MIN_PARK_NS, MAX_PARK_NS);

            final int exitWorkCount = continuous ? -1 : 0;

            int workCount;
            do
            {
                workCount = 0;

                for (int i=0; i < loggables.length; i++)
                {
                    workCount += loggables[i].process();
                }

                idleStrategy.idle(workCount);

            } while (workCount != exitWorkCount);
        }
        catch (IOException ex)
        {
            LangUtil.rethrowUnchecked(ex);
        }
    }

    private boolean nextTimestamp(
        final long timestamp)
    {
        if (timestamp != nextTimestamp)
        {
            nextTimestamp = Math.min(timestamp, nextTimestamp);
            return false;
        }
        else
        {
            nextTimestamp = Long.MAX_VALUE;
            return true;
        }
    }
}
