// TODO: license
package org.reaktivity.command.log.internal.types;

import java.nio.charset.StandardCharsets;
import java.util.function.Consumer;
import org.agrona.BitUtil;
import org.agrona.DirectBuffer;
import org.agrona.MutableDirectBuffer;

public final class TcpAddressFW extends Flyweight {
  private static final int FIELD_SIZE_KIND = BitUtil.SIZE_OF_BYTE;

  private static final int FIELD_OFFSET_KIND = 0;

  public static final int KIND_IPV4_ADDRESS = 1;

  private static final int FIELD_OFFSET_IPV4_ADDRESS = FIELD_OFFSET_KIND + FIELD_SIZE_KIND;

  private static final int FIELD_SIZE_IPV4_ADDRESS = 4;

  public static final int KIND_IPV6_ADDRESS = 2;

  private static final int FIELD_OFFSET_IPV6_ADDRESS = FIELD_OFFSET_KIND + FIELD_SIZE_KIND;

  private static final int FIELD_SIZE_IPV6_ADDRESS = 16;

  public static final int KIND_HOST = 3;

  private static final int FIELD_OFFSET_HOST = FIELD_OFFSET_KIND + FIELD_SIZE_KIND;

  private final OctetsFW ipv4AddressRO = new OctetsFW();

  private final OctetsFW ipv6AddressRO = new OctetsFW();

  private final StringFW hostRO = new StringFW();

  public OctetsFW ipv4Address() {
    return ipv4AddressRO;
  }

  public OctetsFW ipv6Address() {
    return ipv6AddressRO;
  }

  public StringFW host() {
    return hostRO;
  }

  public int kind() {
    return buffer().getByte(offset() + FIELD_OFFSET_KIND) & 0xFF;
  }

  @Override
  public TcpAddressFW tryWrap(DirectBuffer buffer, int offset, int maxLimit) {
    super.wrap(buffer, offset, maxLimit);
    switch (kind()) {
      case KIND_IPV4_ADDRESS: {
        if (null == ipv4AddressRO.tryWrap(buffer, offset + FIELD_OFFSET_IPV4_ADDRESS, offset + FIELD_OFFSET_IPV4_ADDRESS + 4)) {
          return null;
        }
        break;
      }
      case KIND_IPV6_ADDRESS: {
        if (null == ipv6AddressRO.tryWrap(buffer, offset + FIELD_OFFSET_IPV6_ADDRESS, offset + FIELD_OFFSET_IPV6_ADDRESS + 16)) {
          return null;
        }
        break;
      }
      case KIND_HOST: {
        if (null == hostRO.tryWrap(buffer, offset + FIELD_OFFSET_HOST, maxLimit)) {
          return null;
        }
        break;
      }
      default: {
        break;
      }
    }
    if (limit() > maxLimit) {
      return null;
    }
    return this;
  }

  @Override
  public TcpAddressFW wrap(DirectBuffer buffer, int offset, int maxLimit) {
    super.wrap(buffer, offset, maxLimit);
    switch (kind()) {
      case KIND_IPV4_ADDRESS: {
        ipv4AddressRO.wrap(buffer, offset + FIELD_OFFSET_IPV4_ADDRESS, offset + FIELD_OFFSET_IPV4_ADDRESS + 4);
        break;
      }
      case KIND_IPV6_ADDRESS: {
        ipv6AddressRO.wrap(buffer, offset + FIELD_OFFSET_IPV6_ADDRESS, offset + FIELD_OFFSET_IPV6_ADDRESS + 16);
        break;
      }
      case KIND_HOST: {
        hostRO.wrap(buffer, offset + FIELD_OFFSET_HOST, maxLimit);
        break;
      }
      default: {
        break;
      }
    }
    checkLimit(limit(), maxLimit);
    return this;
  }

  @Override
  public int limit() {
    switch (kind()) {
      case KIND_IPV4_ADDRESS: {
        return ipv4Address().limit();
      }
      case KIND_IPV6_ADDRESS: {
        return ipv6Address().limit();
      }
      case KIND_HOST: {
        return host().limit();
      }
      default: {
        return offset();
      }
    }
  }

  @Override
  public String toString() {
    switch (kind()) {
      case KIND_IPV4_ADDRESS: {
        return String.format("TCPADDRESS [ipv4Address=%s]", ipv4Address());
      }
      case KIND_IPV6_ADDRESS: {
        return String.format("TCPADDRESS [ipv6Address=%s]", ipv6Address());
      }
      case KIND_HOST: {
        return String.format("TCPADDRESS [host=%s]", hostRO.asString());
      }
      default: {
        return String.format("TCPADDRESS [unknown]");
      }
    }
  }

  public static final class Builder extends Flyweight.Builder<TcpAddressFW> {
    private final OctetsFW.Builder ipv4AddressRW = new OctetsFW.Builder();

    private final OctetsFW.Builder ipv6AddressRW = new OctetsFW.Builder();

    private final StringFW.Builder hostRW = new StringFW.Builder();

    public Builder() {
      super(new TcpAddressFW());
    }

    private Builder kind(int value) {
      buffer().putByte(offset() + FIELD_OFFSET_KIND, (byte)(value & 0xFF));
      return this;
    }

    private OctetsFW.Builder ipv4Address() {
      int newLimit = offset() + FIELD_OFFSET_IPV4_ADDRESS + FIELD_SIZE_IPV4_ADDRESS;
      checkLimit(newLimit, maxLimit());
      return ipv4AddressRW.wrap(buffer(), offset() + FIELD_OFFSET_IPV4_ADDRESS, newLimit);
    }

    public Builder ipv4Address(Consumer<OctetsFW.Builder> mutator) {
      kind(KIND_IPV4_ADDRESS);
      OctetsFW.Builder ipv4Address = ipv4Address();
      mutator.accept(ipv4Address);
      limit(ipv4Address.build().limit());
      return this;
    }

    private OctetsFW.Builder ipv6Address() {
      int newLimit = offset() + FIELD_OFFSET_IPV6_ADDRESS + FIELD_SIZE_IPV6_ADDRESS;
      checkLimit(newLimit, maxLimit());
      return ipv6AddressRW.wrap(buffer(), offset() + FIELD_OFFSET_IPV6_ADDRESS, newLimit);
    }

    public Builder ipv6Address(Consumer<OctetsFW.Builder> mutator) {
      kind(KIND_IPV6_ADDRESS);
      OctetsFW.Builder ipv6Address = ipv6Address();
      mutator.accept(ipv6Address);
      limit(ipv6Address.build().limit());
      return this;
    }

    private StringFW.Builder host() {
      int newLimit = maxLimit();
      checkLimit(newLimit, maxLimit());
      return hostRW.wrap(buffer(), offset() + FIELD_OFFSET_HOST, newLimit);
    }

    public Builder host(String value) {
      if (value == null) {
        limit(offset() + FIELD_OFFSET_HOST);
      } else {
        kind(KIND_HOST);
        StringFW.Builder host = host();
        host.set(value, StandardCharsets.UTF_8);
        limit(host.build().limit());
      }
      return this;
    }

    public Builder wrap(MutableDirectBuffer buffer, int offset, int maxLimit) {
      super.wrap(buffer, offset, maxLimit);
      return this;
    }
  }
}
