// TODO: license
package org.reaktivity.command.log.internal.types;

import java.util.function.Consumer;
import org.agrona.DirectBuffer;
import org.agrona.MutableDirectBuffer;
import org.agrona.concurrent.UnsafeBuffer;

public abstract class Flyweight {
  private static byte[] EMPTY_BYTES = new byte[0];

  private DirectBuffer buffer;

  private int offset;

  private int maxLimit;

  private UnsafeBuffer compareBuffer = new UnsafeBuffer(EMPTY_BYTES);

  public final int offset() {
    return offset;
  }

  public final DirectBuffer buffer() {
    return buffer;
  }

  public abstract int limit();

  public final int sizeof() {
    return limit() - offset();
  }

  protected final int maxLimit() {
    return maxLimit;
  }

  public Flyweight tryWrap(DirectBuffer buffer, int offset, int maxLimit) {
    if (offset > maxLimit) {
      return null;
    }
    this.buffer = buffer;
    this.offset = offset;
    this.maxLimit = maxLimit;
    return this;
  }

  public Flyweight wrap(DirectBuffer buffer, int offset, int maxLimit) {
    if (offset > maxLimit) {
      final String msg = String.format("offset=%d is beyond maxLimit=%d", offset, maxLimit);
      throw new IndexOutOfBoundsException(msg);
    }
    this.buffer = buffer;
    this.offset = offset;
    this.maxLimit = maxLimit;
    return this;
  }

  protected static final void checkLimit(int limit, int maxLimit) {
    if (limit > maxLimit) {
      final String msg = String.format("limit=%d is beyond maxLimit=%d", limit, maxLimit);
      throw new IndexOutOfBoundsException(msg);
    }
  }

  @Override
  public boolean equals(Object obj) {
    if (this == obj) {
      return true;
    } else if (obj == null || !(obj instanceof Flyweight)) {
      return false;
    } else {
      Flyweight that = (Flyweight) obj;
      compareBuffer.wrap(buffer, offset, sizeof());
      that.compareBuffer.wrap(that.buffer, that.offset, that.sizeof());
      return compareBuffer.equals(that.compareBuffer);
    }
  }

  @Override
  public int hashCode() {
    int result = 1;
    for (int i = offset; i < limit(); i++) {
      result = 31 * result + buffer.getByte(i);
    }
    return result;
  }

  @FunctionalInterface
  public interface Visitor<T> {
    T visit(DirectBuffer buffer, int offset, int maxLimit);
  }

  public abstract static class Builder<T extends Flyweight> {
    private final T flyweight;

    private MutableDirectBuffer buffer;

    private int offset;

    private int limit;

    private int maxLimit;

    protected Builder(T flyweight) {
      this.flyweight = flyweight;
    }

    public final int limit() {
      return limit;
    }

    public final int maxLimit() {
      return maxLimit;
    }

    public T build() {
      flyweight.wrap(buffer, offset, limit);
      return flyweight;
    }

    public Builder<T> rewrap() {
      this.limit = this.offset;
      return this;
    }

    protected final T flyweight() {
      return flyweight;
    }

    protected final MutableDirectBuffer buffer() {
      return buffer;
    }

    protected final int offset() {
      return offset;
    }

    protected final void limit(int limit) {
      this.limit = limit;
    }

    public Builder<T> wrap(MutableDirectBuffer buffer, int offset, int maxLimit) {
      this.buffer = buffer;
      this.offset = offset;
      this.limit = offset;
      this.maxLimit = maxLimit;
      return this;
    }

    public <E> Builder<T> iterate(Iterable<E> iterable, Consumer<E> action) {
      iterable.forEach(action);
      return this;
    }

    @FunctionalInterface
    public interface Visitor {
      int visit(MutableDirectBuffer buffer, int offset, int maxLimit);
    }
  }
}
