// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.audio;

/**
 * The {@link BitsPerSample} enumeration defines some common bits/sample per
 * channel constants.
 */
public enum BitsPerSample {

	/**
	 * Low resolution of 8 bits/sample and channel.
	 */
	LOW_RES(8),

	/**
	 * High resolution of 16 bits/sample and channel.
	 */
	HIGH_RES(16);

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private int _bitsPerSample;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	private BitsPerSample( int aResolution ) {
		_bitsPerSample = aResolution;
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Returns the bits/sample per channel.
	 * 
	 * @return The bits/sample for this {@link BitsPerSample}.
	 */
	public int getBitCount() {
		return _bitsPerSample;
	}

	/**
	 * Returns the bytes/sample per channel.
	 * 
	 * @return The bytes/sample for this {@link BitsPerSample}.
	 */
	public int getByteCount() {
		return _bitsPerSample / 8;
	}

	/**
	 * Retrieves the lowest bits/sample resolution.
	 * 
	 * @return The lowest resolution.
	 */
	public static BitsPerSample getLowestResolution() {
		return LOW_RES;
	}

	/**
	 * Retrieves the lowest bits/sample resolution.
	 * 
	 * @return The highest resolution.
	 */
	public static BitsPerSample getHigestResolution() {
		return HIGH_RES;
	}

	/**
	 * Retrieves the next higher bits/sample or null if it is already the
	 * highest bits/sample.
	 * 
	 * @return The next higher bits/sample or null if already the highest bits /
	 *         sample.
	 */
	public BitsPerSample getNextHigherResolution() {
		return toNextHigherResolution( this );
	}

	/**
	 * Retrieves the previous lower bits/sample or null if it is already the
	 * lowest bits/sample.
	 * 
	 * @return The previous lower bits/sample or null if already the lowest
	 *         bits/sample.
	 */
	public BitsPerSample getPreviousLowerResolution() {
		return toPreviousLowerResolution( this );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Retrieves the next higher bits/sample or null if it is already the
	 * highest bits/sample.
	 * 
	 * @param aResolution The bits/sample for which to get the next higher one.
	 * 
	 * @return The next higher bits/sample or null if already the highest
	 *         bits/sample.
	 */
	private static BitsPerSample toNextHigherResolution( BitsPerSample aResolution ) {
		for ( int i = 0; i < values().length - 1; i++ ) {
			if ( aResolution == values()[i] ) {
				return values()[i + 1];
			}
		}
		return null;
	}

	/**
	 * Retrieves the previous lower bits/sample or null if it is already the
	 * lowest bits/sample.
	 * 
	 * @param aResolution The bits/sample for which to get the previous lower
	 *        one.
	 * 
	 * @return The previous lower bits/sample or null if already the lowest
	 *         bits/sample.
	 */
	private static BitsPerSample toPreviousLowerResolution( BitsPerSample aResolution ) {
		for ( int i = 1; i < values().length; i++ ) {
			if ( aResolution == values()[i] ) {
				return values()[i - 1];
			}
		}
		return null;
	}
}
